package com.candata.login.zoo;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.stream.Stream;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceEvent;
import org.osgi.framework.startlevel.BundleStartLevel;
import org.osgi.framework.startlevel.FrameworkStartLevel;

import com.candata.login.zoo.provision.interfaces.ProgressMonitor;

public class Launcher
{
	private Runnable mainThread;

	public boolean run(BundleContext ctx, ProgressMonitor monitor)
	{
		monitor.configure("Launching application\u2026");
		try
		{
			CountDownLatch waitForMainThread = new CountDownLatch(1);
			listenForMain(ctx, waitForMainThread);
			logger.log(Level.INFO, "Setting Run Level " + UI_START_LEVEL);
			FrameworkStartLevel startLevel = ctx.getBundle(0).adapt(FrameworkStartLevel.class);
			CountDownLatch waitForStart = new CountDownLatch(1);
			startLevel.setStartLevel(UI_START_LEVEL, e -> waitForStart.countDown());
			boolean status = waitForStart.await(10, TimeUnit.SECONDS);
			if (!status)
			{

			}
			boolean foundMainThread = waitForMainThread.await(10, TimeUnit.SECONDS);
			if (!foundMainThread)
			{
				System.out.println("bundle count: " + ctx.getBundles().length);
				Stream.of(ctx.getBundles())
						.sorted(Comparator.comparing(Bundle::getSymbolicName))
						.forEach(b -> {
							System.out.println("bundle: " + b);
							System.out.println("   start level: " + b.adapt(BundleStartLevel.class).getStartLevel());
							System.out.println("   state: " + b.getState());
						});
				//				System.out.println(listUnstartedServices(ctx));
				System.err.println("No service registered as Runnable with property main.thread=true");
				return false;
			}
			mainThread.run();
			return true;
		}
		catch (Throwable e)
		{
			e.printStackTrace();
			error("Fatal error. Installed Bundles");
			dumpBundles(ctx);
			error("Unexpected error in the run body: %s", e);
			throw new RuntimeException(e);
		}
	}

	//	private static String listUnstartedServices(BundleContext ctx)
	//	{
	//		ServiceComponentRuntime ser = ctx.getService(ctx.getServiceReference(ServiceComponentRuntime.class));
	//		Collection<ComponentDescriptionDTO> comps = ser.getComponentDescriptionDTOs();
	//		StringBuilder buf = new StringBuilder();
	//		for (ComponentDescriptionDTO c : comps)
	//		{
	//			if (c.factory != null)
	//			{
	//				checkFactory(c, ctx.getBundle(c.bundle.id).getBundleContext(), buf);
	//			}
	//			else
	//			{
	//				for (ComponentConfigurationDTO d : ser.getComponentConfigurationDTOs(c))
	//				{
	//					if (d.state != ComponentConfigurationDTO.ACTIVE && d.state != ComponentConfigurationDTO.SATISFIED)
	//					{
	//						Map<String, Object> prop = c.properties;
	//						String factoryName = c.factory;
	//
	//						if (prop.get("candata.optional.service") == null)
	//						{
	//							addMissingRef(d, buf);
	//						}
	//					}
	//				}
	//			}
	//		}
	//		return buf.toString();
	//	}
	//
	//	private static void addMissingRef(ComponentConfigurationDTO c, StringBuilder buf)
	//	{
	//		StringBuilder tmp = new StringBuilder();
	//		for (UnsatisfiedReferenceDTO r : c.unsatisfiedReferences)
	//		{
	//			tmp.append(
	//					"\n\tMissing " + r.name + " " + Stream.of(r.targetServices).map(ServiceReferenceDTO::toString).collect(Collectors.joining("\n")));
	//		}
	//
	//		if (tmp.length() > 0)
	//		{
	//			tmp.insert(0, c.description.name + " is not satisfied ");
	//			buf.append(tmp);
	//			buf.append("\n");
	//		}
	//	}
	//
	//	private static void checkFactory(ComponentDescriptionDTO c, BundleContext ctx, StringBuilder buf)
	//	{
	//		try
	//		{
	//			Collection<ServiceReference<ComponentFactory>> refs = ctx.getServiceReferences(ComponentFactory.class,
	//					"(" + ComponentConstants.COMPONENT_FACTORY + "=" + c.factory + ")");
	//			if (refs.size() == 1)
	//			{
	//				return;
	//			}
	//
	//			buf.append("Missing factory service " + c.factory + "\n");
	//			for (ReferenceDTO r : c.references)
	//			{
	//				if (!r.cardinality.equals("0..1") && !r.cardinality.equals("0..n")) // Exclude multiple and optional
	//				{
	//					ServiceReference<?>[] exists = ctx.getServiceReferences(r.interfaceName, r.target);
	//					if (exists == null)
	//					{
	//						buf.append("\t" + r.interfaceName);
	//						if (r.target != null)
	//						{
	//							buf.append(" " + r.target);
	//						}
	//						buf.append("\n");
	//					}
	//				}
	//			}
	//		}
	//		catch (InvalidSyntaxException e)
	//		{
	//			throw new RuntimeException(e);
	//		}
	//
	//	}
	//
	protected void listenForMain(BundleContext ctx, CountDownLatch waitForMainThread) throws InvalidSyntaxException
	{
		ctx.addServiceListener(event -> {
			if (event.getType() == ServiceEvent.REGISTERED)
			{
				mainThread = (Runnable) ctx.getService(event.getServiceReference());
				if (mainThread != null)
				{
					waitForMainThread.countDown();
				}
			}
		}, "(&(objectclass=java.lang.Runnable)(main.thread=true))");

	}

	private void dumpBundles(BundleContext ctx)
	{
		Stream.of(ctx.getBundles())
				.forEach(bundle -> logger.log(Level.SEVERE, bundle.getSymbolicName()));
	}

	private List<String> getMetaInfServices(ClassLoader loader, String factory)
			throws IOException
	{
		Enumeration<URL> e = loader.getResources("META-INF/services/" + factory);
		List<String> factories = new ArrayList<>();

		while (e.hasMoreElements())
		{
			URL url = e.nextElement();

			InputStream in = null;
			BufferedReader rdr = null;
			String line;
			try
			{
				in = url.openStream();
				rdr = new BufferedReader(new InputStreamReader(in, "UTF-8"));
				while ((line = rdr.readLine()) != null)
				{
					line = line.trim();
					if (!line.startsWith("#") && line.length() > 0)
					{
						factories.add(line);
					}
				}
			}
			finally
			{
				if (rdr != null)
				{
					rdr.close();
				}
				if (in != null)
				{
					in.close();
				}
			}
		}
		return factories;
	}

	private String translateToMessage(BundleException e)
	{
		switch (e.getType())
		{
		case BundleException.ACTIVATOR_ERROR:
			Throwable t = e.getCause();
			StackTraceElement[] stackTrace = t.getStackTrace();
			if (stackTrace == null || stackTrace.length == 0)
			{
				return "activator error " + t.getMessage();
			}
			StackTraceElement top = stackTrace[0];
			return "activator error " + t.getMessage() + " from: " + top.getClassName() + ":" + top.getMethodName()
					+ "#" + top.getLineNumber();

		case BundleException.DUPLICATE_BUNDLE_ERROR:
		case BundleException.RESOLVE_ERROR:
		case BundleException.INVALID_OPERATION:
		case BundleException.MANIFEST_ERROR:
		case BundleException.NATIVECODE_ERROR:
		case BundleException.STATECHANGE_ERROR:
		case BundleException.UNSUPPORTED_OPERATION:
		case BundleException.UNSPECIFIED:
		default:
			return e.getMessage();
		}
	}

	private void message(String prefix, String string, Object[] objects)
	{
		Throwable e = null;

		StringBuilder sb = new StringBuilder();
		int n = 0;
		sb.append(prefix);
		for (int i = 0; i < string.length(); i++)
		{
			char c = string.charAt(i);
			if (c == '%')
			{
				c = string.charAt(++i);
				switch (c)
				{
				case 's':
					if (n < objects.length)
					{
						Object o = objects[n++];
						if (o instanceof Throwable)
						{
							e = (Throwable) o;
							if (o instanceof BundleException)
							{
								sb.append(translateToMessage((BundleException) o));
							}
							else if (o instanceof InvocationTargetException)
							{
								Throwable t = (InvocationTargetException) o;
								sb.append(t.getMessage());
								e = t;
							}
							else
							{
								sb.append(e.getMessage());
							}
						}
						else
						{
							sb.append(o);
						}
					}
					else
					{
						sb.append("<no more arguments>");
					}
					break;

				default:
					sb.append(c);
				}
			}
			else
			{
				sb.append(c);
			}
		}
		System.out.println(sb);
		if (e != null)
		{
			e.printStackTrace(System.out);
		}
		System.out.flush();
	}

	protected void error(String msg, Throwable t)
	{
		logger.log(Level.SEVERE, "Exiting", t);
		System.exit(1);
	}

	public void error(String msg, Object... objects)
	{
		message("! ", msg, objects);
		logger.log(Level.SEVERE, "Exiting: " + msg);
		System.exit(0);
	}

	public static final int UI_START_LEVEL = 10;
	public static final int START_LEVEL = 6;
	protected static java.util.logging.Logger logger = java.util.logging.Logger.getLogger("Candata");
}
