package com.candata.login.zoo;

import static com.candata.login.oauth.oauth2.Constants.SUPPORT_EMAIL_DOMAIN;
import static com.candata.login.utils.StringUtils.EMPTY;
import static com.candata.login.zoo.utils.DisplayThread.async;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.json.JSONArray;
import org.json.JSONObject;

import com.candata.login.oauth.Authenticator;
import com.candata.login.oauth.beans.Authentication;
import com.candata.login.oauth.beans.Authentication.SupportAuthentications;
import com.candata.login.oauth.beans.Environment;
import com.candata.login.oauth.oauth2.beans.OAuthProperties;
import com.candata.login.oauth.oauth2.beans.ValidationResponse;
import com.candata.login.oauth.support.oauth2.ValidateToken;
import com.candata.login.oauth.support.oauth2.beans.OAuthSupportProperties;
import com.candata.login.oauth.users.Application;
import com.candata.login.oauth.users.Company;
import com.candata.login.oauth.users.User;
import com.candata.login.zoo.oauth2.beans.LastLogin;
import com.candata.login.zoo.provision.interfaces.Progress;
import com.candata.login.zoo.provision.interfaces.ProgressMonitor;
import com.candata.login.zoo.servers.Instance;
import com.candata.login.zoo.users.Companies;
import com.candata.login.zoo.users.UserValidation;
import com.candata.login.zoo.users.Users;
import com.candata.login.zoo.utils.DisplayThread;
import com.candata.login.zoo.widgets.ImageWidget;
import com.candata.login.zoo.widgets.ProgressIndicator;
import com.candata.login.zoo.widgets.listeners.MoveShellListener;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Maybe;
import io.reactivex.rxjava3.schedulers.Schedulers;
import io.reactivex.rxjava3.subjects.MaybeSubject;

public class Login
{

	protected static java.util.logging.Logger logger = java.util.logging.Logger.getLogger("Candata");

	private LoginShell shell;
	private String user;
	//	BundleContext ctx;
	MaybeSubject<Authentication> authenticated;

	/**
	 * Launch the application.
	 * @param args
	 */
	public static void main(String args[])
	{
		try
		{
			Display display = Display.getDefault();
			LoginShell shell = new LoginShell(display);
			shell.open();
			shell.layout();
			while (!shell.isDisposed())
			{
				if (!display.readAndDispatch())
				{
					display.sleep();
				}
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}

	/**
	 * Create the shell.
	 * @param display
	 */

	public Login(Display display)
	{
		shell = new LoginShell(display);
		launch(shell);
		progressMonitor = getProgressMonitor(shell);
		shell.layout();
		shell.setText("Candata Global");
		shell.positionShell();
		display.asyncExec(() -> shell.positionShell());
		shell.open();
	}

	public Shell getLoginShell()
	{
		return shell;
	}

	protected void setAuthenticationCallback(MaybeSubject<Authentication> authenticated)
	{
		this.authenticated = authenticated;
	}

	public ProgressMonitor getProgressMonitor()
	{
		return progressMonitor;
	}

	protected ProgressMonitor getProgressMonitor(LoginShell shell)
	{
		return new ProgressMonitor() {
			@Override
			public void update(Progress progress)
			{
				DisplayThread.async(shell, () -> updateProvision(shell, progress));
			}

			@Override
			public void configure(String task, int total)
			{
				DisplayThread.async(shell, () -> startProvision(shell, task, total));
			}

			@Override
			public void complete()
			{
				DisplayThread.async(shell, () -> shell.dispose());
			}
		};
	}

	protected void launch(LoginShell shell)
	{
		//		createContents();
		configureOKButton(shell);
		configureCancelButton(shell);
		configureEmailControls(shell);
		configureSupportCompanyCombo(shell);
		setEmailState(shell);
		//		pack();

		hideStatus(shell);
		// login
		setBestFocusControl(shell);
		MoveShellListener.add(shell);
		com.candata.login.zoo.os.SWT.adjustDPI(shell);
		shell.setTabList(new Control[] { shell.contentComposite, shell.buttonComposite, shell.titleComposite });

		checkLastLogin(shell);
	}

	private void handleInitialEmail(LoginShell shell)
	{
		shell.emailText.setEnabled(false);
		shell.okCancelButtons.getOKButton().setEnabled(false);
		shell.verifyingEmailImageWidget.setVisible(true);
		shell.verifyingEmailLabel.setVisible(true);
		shell.emailResultComposite.setVisible(true);
		shell.statusErrorComposite.setVisible(false);
		shell.emailResultCompositeLayout.topControl = shell.verifyingEmailComposite;
		shell.emailComposite.layout(true, true);

		if (shell.emailText.getText().endsWith(SUPPORT_EMAIL_DOMAIN))
		{
			shell.loginEmailLabel.setText(shell.emailText.getText());
			checkSupportUser(shell);
			return;
		}
		UserValidation.isValid(shell.emailText.getText())
				.doOnSuccess(u -> validUser(shell, u))
				.doOnComplete(() -> invalidUser(shell))
				.doOnError(t -> logger.log(Level.SEVERE, "error processing email address", t))
				.doOnError(e -> errorValidatingUser(shell, e))
				.onErrorComplete()
				.subscribe();
	}

	protected void setEmailState(LoginShell shell)
	{
		shell.sl_contentComposite.topControl = shell.emailComposite;
		shell.leftButtonCompositeLayout.topControl = shell.proxyLink;
		//		okCancelButtons.getCancelButton().setVisible(false);
		shell.okCancelButtons.setAsNext();
		handleOKState(shell);
		shell.emailResultComposite.setVisible(false);
		shell.emailResultCompositeLayout.topControl = shell.verifyingEmailComposite;

		shell.proxyLink.setVisible(true);
		shell.buttonComposite.layout(true, true);

		shell.emailText.setEnabled(true);
		shell.contentComposite.getParent().layout(true, true);

		async(shell.emailText, () -> {
			shell.emailText.setFocus();
			shell.emailText.selectAll();
		});
	}

	protected void checkLastLogin(LoginShell shell)
	{
		shell.rememberMeButton.setSelection(LastLogin.getRememberMe());
		System.setProperty(OAuthProperties.REMEMBER_ME_KEY, LastLogin.getRememberMe().toString());
		LastLogin.getEmail()
				.ifPresent(email -> {
					shell.emailText.setText(email);
					if (shell.okCancelButtons.getOKButton().isEnabled())
					{
						handleInitialEmail(shell);
					}
				});
	}

	protected void setBestFocusControl(LoginShell shell)
	{
		if (shell.loginEmailLabel.getText().isEmpty())
		{
			shell.loginEmailLabel.setFocus();
			return;
		}
		shell.userServerCombo.setFocus();
	}

	protected Image getImage(String name)
	{
		return images.computeIfAbsent(name, this::doGetImage);
	}

	protected Image doGetImage(String name)
	{
		try (InputStream in = getImageInputStream(name))
		{
			return (in == null) ? null : new Image(Display.getDefault(), in);
		}
		catch (java.io.IOException e)
		{
		}
		return null;
	}

	private ImageData[] getAnimatedImage(String name)
	{
		if (name.endsWith(".pngz"))
		{
			try
			{
				return loadPNG(name);
			}
			catch (IOException e)
			{
			}
		}
		try (InputStream in = getImageInputStream(name))
		{
			return (in == null) ? null : new ImageLoader().load(in);
		}
		catch (java.io.IOException e)
		{
		}
		return null;
	}

	private ImageData[] loadPNG(String name) throws IOException
	{
		ZipInputStream zis = new ZipInputStream(getImageInputStream(name));
		Map<String, ImageData> imageMap = new HashMap<>();
		// Loop through each entry in the ZIP
		ZipEntry entry;
		while ((entry = zis.getNextEntry()) != null)
		{
			if (!entry.isDirectory() && !entry.getName().startsWith("__"))
			{
				ImageData[] imageData = new ImageLoader().load(zis);
				imageMap.put(entry.getName(), imageData[0]);
			}
		}
		zis.close();
		return imageMap.entrySet()
				.stream()
				.sorted(Comparator.comparing(Entry::getKey))
				.map(Entry::getValue)
				.toArray(ImageData[]::new);
	}

	protected InputStream getImageInputStream(String name)
	{
		if (Login.class.getClassLoader().getResourceAsStream(name) != null)
		{
			return Login.class.getClassLoader().getResourceAsStream(name);
		}
		return Login.class.getClassLoader().getResourceAsStream("images/" + name);
	}

	protected void exitApplication(LoginShell shell)
	{
		System.setProperty(LoginStarter.LOGIN_CANCELLED_KEY, "true");
		//		logger = null;
		shell.close();
		shell.dispose();
		//		serverBaseURL = null;
		//		hostnameVerifier = null;
	}

	private void hideStatus(LoginShell shell)
	{
		//		userNameErrorLabel.setVisible(false);
		//		loginMessageLabel.setVisible(false);
		//		imageWidget.stop();
		shell.loginMessageLabel.getParent().layout(true, true);
	}

	protected void resetStatus(LoginShell shell)
	{
		shell.connectedStatusComposite.setVisible(false);
		// GridData gd_status = (GridData)
		// connectedStatusComposite.getLayoutData();
		// gd_status.exclude = true;
		// connectedStatusComposite.setLayoutData(gd_status);
		shell.connectedStatusComposite.getParent().layout(true, true);
		shell.connectedStatusComposite.getShell().redraw();
	}

	private void configureEmailControls(LoginShell shell)
	{
		shell.loginEmailLabel.addMouseListener(MouseListener.mouseDoubleClickAdapter(e -> setEmailState(shell)));
		shell.switchUserLabel.addMouseListener(MouseListener.mouseDownAdapter(e -> setEmailState(shell)));
		Listener switchUserKeyListener = event -> {
			if (event.widget.isDisposed() || event.widget.getDisplay().isDisposed())
			{
				return;
			}
			if (shell.switchUserLabel.isDisposed() || !shell.switchUserLabel.isVisible() || !shell.switchUserLabel.isEnabled())
			{
				return;
			}
			Shell s = event.widget.getDisplay().getActiveShell();
			if ((s == null) || s.isDisposed() || (s != shell.switchUserLabel.getShell()))
			{
				return;
			}
			if (event.keyCode == SWT.F2)
			{
				setEmailState(shell);
			}
		};
		shell.switchUserLabel.getDisplay().addFilter(SWT.KeyDown, switchUserKeyListener);
		shell.switchUserLabel.addDisposeListener(e -> shell.switchUserLabel.getDisplay().removeFilter(SWT.KeyDown, switchUserKeyListener));
		shell.emailText.addModifyListener(event -> handleOKState(shell));
	}

	private void handleOKState(LoginShell shell)
	{
		shell.leftButtonComposite.setVisible(!shell.emailText.getText().endsWith(SUPPORT_EMAIL_DOMAIN));
		shell.okCancelButtons.getOKButton().setEnabled(EMAIL_PATTERN.matcher(shell.emailText.getText()).matches());
	}

	private void validSupportUser(LoginShell shell, ValidationResponse response, JSONArray allUsers)
	{
		if (!shell.emailText.isDisposed())
		{
			JSONArray namespaces = new JSONArray(response.getMessage());
			supportAuthentications = response.getAuthentications();
			JSONObject payload = supportAuthentications.getProduction().getIdTokenPayload();
			User supportUser = new User(payload.getString("email"));
			supportUser.setFirstName(payload.optString("given_name"));
			supportUser.setLastName(payload.optString("family_name"));
			currentUser = supportUser;
			shell.btnProduction.setEnabled(false);
			shell.btnTest.setEnabled(false);
			shell.supportImpersonateCombo.setEnabled(false);
			shell.emailResultComposite.setVisible(true);
			shell.emailResultCompositeLayout.topControl = shell.verifiedEmailComposite;
			shell.emailComposite.layout(true, true);
			//							handleServers(user);
			Display.getDefault().timerExec(500, () -> {
				if (!shell.emailText.isDisposed())
				{
					parseCompaniesAndUsers(shell, namespaces, allUsers);
					shell.proxyLink.setVisible(false);
					shell.loginEmailLabel.setText(shell.emailText.getText());
					shell.statusStackLayout.topControl = shell.validSupportUserComposite;
					shell.statusErrorComposite.layout();

					shell.sl_contentComposite.topControl = shell.loginFrameComposite;
					shell.contentComposite.getParent().layout(true, true);
					shell.loginTypeStackCompositeLayout.topControl = shell.supportLoginComposite;
					shell.contentComposite.getParent().layout(true, true);
					shell.buttonComposite.layout(true, true);
					shell.okCancelButtons.setAsOK();
					Button okButton = shell.okCancelButtons.getOKButton();
					okButton.setEnabled(true);
				}
			});
		}
	}

	private void parseCompaniesAndUsers(LoginShell shell, JSONArray namespaces, JSONArray allUsers)
	{
		Instance.get(namespaces)
				.toList()
				.doOnSuccess(n -> this.instances = n)
				.ignoreElement()
				.andThen(Users.get(allUsers)
						.toList()
						.doOnSuccess(users -> this.users = users)
						.ignoreElement())
				.andThen(Companies.get(allUsers)
						.sorted(Comparator.comparing(Company::getName))
						.toList()
						.doOnSuccess(companies -> this.companies = companies)
						.doOnSuccess(c -> async(shell.supportServerCombo, () -> configureCompanies(shell)))
						.ignoreElement())
				.subscribe();
	}

	private void configureCompanies(LoginShell shell)
	{
		String[] companyLabels = companies
				.stream()
				.map(Company::getName)
				.distinct()
				.toArray(String[]::new);
		shell.supportServerCombo.setItems(companyLabels);
		if (companies.size() > 1)
		{
			async(shell.supportServerCombo, shell.supportServerCombo::setFocus);
		}
		if (LastLogin.getEmail().equals(Optional.of(currentUser.getEmail())))
		{
			Optional<String> lastCompany = LastLogin.getCompany();
			if (lastCompany.isPresent())
			{
				String company = lastCompany.get();
				String[] parts = company.split(" ");
				boolean isTest = (parts.length > 1 && parts[1].endsWith("(test)"));
				shell.btnProduction.setSelection(!isTest);
				shell.btnTest.setSelection(isTest);
				shell.supportServerCombo.select(shell.supportServerCombo.indexOf(parts[0]));
				handleCompanySelection(shell);
				return;
			}
		}
		shell.userServerCombo.select(0);
	}

	private void validUser(LoginShell shell, User user)
	{
		currentUser = user;
		Display.getDefault().asyncExec(() -> {
			if (!shell.emailText.isDisposed())
			{
				shell.emailResultComposite.setVisible(true);
				shell.emailResultCompositeLayout.topControl = shell.verifiedEmailComposite;
				shell.emailComposite.layout(true, true);
				handleCompanies(shell, user);
				Display.getDefault().timerExec(500, () -> {
					if (!shell.emailText.isDisposed())
					{
						shell.proxyLink.setVisible(false);
						shell.loginEmailLabel.setText(shell.emailText.getText());
						shell.sl_contentComposite.topControl = shell.loginFrameComposite;
						shell.loginTypeStackCompositeLayout.topControl = shell.loginComposite;

						shell.contentComposite.getParent().layout(true, true);
						shell.buttonComposite.layout(true, true);
						shell.okCancelButtons.setAsOK();
						Button okButton = shell.okCancelButtons.getOKButton();
						okButton.setEnabled(true);
						shell.leftButtonCompositeLayout.topControl = shell.rememberMeButton;
						shell.leftButtonComposite.layout(true, true);
					}
				});
			}
		});
	}

	protected String getCompanyLabel(Company company)
	{
		StringBuilder label = new StringBuilder(company.getName());
		if (company.getEnvironment() == Environment.Test)
		{
			label.append(" (");
			label.append("test");
			label.append(")");
		}
		return label.toString();
	}

	protected String getUserLabel(User user)
	{
		if (user.isEmpty())
		{
			return "<None>";
		}
		StringBuilder label = new StringBuilder();
		label.append(user.getFirstName());
		label.append(" ");
		label.append(user.getLastName());
		label.append(" <");
		label.append(user.getEmail());
		label.append(">");
		return label.toString();
	}

	protected String getInstanceLabel(Instance instance, Instance newest)
	{
		if (instance.id().isEmpty())
		{
			String value = newest.id().replace(BUILD_PREFIX, EMPTY);
			return "Latest (" + value + ")";
		}
		return instance.id().replace(BUILD_PREFIX, EMPTY);
	}

	protected void handleCompanies(LoginShell shell, User user)
	{
		user.getCompanies()
				.sort(Comparator.comparing(this::getCompanyLabel));
		String[] companies = user.getCompanies()
				.stream()
				.map(this::getCompanyLabel)
				.distinct()
				.toArray(String[]::new);
		shell.userServerCombo.setItems(companies);
		shell.userServerStackedCompositeLayout.topControl = (user.getCompanies().size() == 1) ? shell.userServerLabelComposite
				: shell.userServerCombo;
		shell.userServerLabel.setText(user.getCompanies().get(0).getName());
		if (companies.length > 1)
		{
			async(shell.userServerCombo, 700, shell.userServerCombo::setFocus);
		}
		if (LastLogin.getEmail().equals(Optional.of(user.getEmail())))
		{
			Optional<String> lastCompany = LastLogin.getCompany();
			if (lastCompany.isPresent())
			{
				shell.userServerCombo.select(shell.userServerCombo.indexOf(lastCompany.get()));
				async(shell.userServerCombo, 700, shell.userServerCombo::setFocus);
				return;
			}
		}
		shell.userServerCombo.select(0);
	}

	private void errorValidatingUser(LoginShell shell, Throwable t)
	{
		emailValidation(shell, ERROR_VALIDATING_EMAIL, t.getMessage());
	}

	private void invalidUser(LoginShell shell)
	{
		emailValidation(shell, INVALID_EMAIL, "");
	}

	private void emailValidation(LoginShell shell, String text, String tooltipMessage)
	{
		Display.getDefault().asyncExec(() -> {
			if (!shell.emailText.isDisposed())
			{
				shell.invalidEmailLabel.setText(text);
				shell.invalidEmailLabel.setToolTipText(tooltipMessage);
				shell.emailText.setEnabled(true);
				shell.emailText.setFocus();
				shell.emailText.selectAll();
				shell.emailResultComposite.setVisible(true);
				shell.emailResultCompositeLayout.topControl = shell.invalidEmailComposite;
				shell.emailComposite.layout(true, true);
			}
		});
	}

	private void configureOKButton(LoginShell shell)
	{
		shell.okCancelButtons.getOKButton().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event)
			{
				if (shell.sl_contentComposite.topControl == shell.emailComposite)
				{
					handleInitialEmail(shell);
					return;
				}
				if (currentUser.isSupport())
				{
					handleSupportLogin(shell);
					return;
				}
				hideStatus(shell);
				shell.cacheClearedComposite.setVisible(false);
				user = shell.loginEmailLabel.getText();
				if (user.trim().isEmpty())
				{
					shell.statusStackLayout.topControl = shell.userNameErrorLabel;
					shell.userNameErrorLabel.setVisible(true);
					shell.loginEmailLabel.setFocus();
				}
				checkUser(shell);
			}
		});
	}

	private void configureCancelButton(LoginShell shell)
	{
		shell.okCancelButtons.getCancelButton().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event)
			{
				authenticated.onComplete();
				exitApplication(shell);
			}
		});
	}

	private void handleSupportLogin(LoginShell shell)
	{
		if (shell.supportServerCombo.getSelectionIndex() == -1)
		{
			return;
		}
		Company company = getSelectedCompany(shell);
		LastLogin.save(shell.loginEmailLabel.getText(), getCompanyLabel(company), true);

		supportAuthentications.setCompany(company);
		if (shell.supportInstanceCombo.getSelectionIndex() != 0)
		{
			supportAuthentications.setRXId(BUILD_PREFIX + shell.supportInstanceCombo.getText());
		}
		if (shell.supportImpersonateCombo.getSelectionIndex() != 0)
		{
			int index = shell.supportImpersonateCombo.getSelectionIndex();
			User impersonate = usersToImpersonate.get(index);
			supportAuthentications.setImpersonatedEmail(impersonate.getEmail());
		}
		provision(supportAuthentications.getAuthentication());
	}

	private int getComboIndex(LoginShell shell, Company company)
	{
		return Arrays.asList(shell.supportServerCombo.getItems())
				.indexOf(company.getName());
	}

	private void configureSupportCompanyCombo(LoginShell shell)
	{
		shell.supportInstanceCombo.setEnabled(false);
		shell.supportServerCombo.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent event)
			{
				handleCompanySelection(shell);
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent event)
			{
				widgetSelected(event);
			}
		});
		shell.supportServerCombo.addModifyListener(event -> {
			if (shell.supportServerCombo.getSelectionIndex() == -1)
			{
				int index = companies.stream()
						.filter(c -> c.getName().equals(shell.supportServerCombo.getText()))
						.findFirst()
						.map(c -> getComboIndex(shell, c))
						.orElse(-1);
				if (index != -1)
				{
					async(shell.supportServerCombo, () -> {
						shell.supportServerCombo.select(index);
						handleCompanySelection(shell);
					});
					return;
				}
				handleCompanySelection(shell);
			}
		});
		shell.supportServerCombo.addFocusListener(new FocusListener() {
			@Override
			public void focusLost(FocusEvent event)
			{
				if (shell.supportServerCombo.getSelectionIndex() == -1)
				{
					int index = companies.stream()
							.filter(c -> c.getName().equals(shell.supportServerCombo.getText()))
							.findFirst()
							.map(c -> getComboIndex(shell, c))
							.orElse(-1);
					async(shell.supportServerCombo, () -> {
						shell.supportServerCombo.select(index);
						if (index == -1)
						{
							shell.supportServerCombo.setText("");
						}
						handleCompanySelection(shell);
					});
				}
			}

			@Override
			public void focusGained(FocusEvent event)
			{
				//				supportServerCombo.set
			}
		});
		shell.btnProduction.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent event)
			{
				if (shell.btnProduction.getSelection())
				{
					handleEnvironmentSelection(shell);
				}
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent event)
			{
				widgetSelected(event);
			}
		});
		shell.btnTest.addSelectionListener(new SelectionListener() {
			@Override
			public void widgetSelected(SelectionEvent event)
			{
				if (shell.btnTest.getSelection())
				{
					handleEnvironmentSelection(shell);
				}
			}

			@Override
			public void widgetDefaultSelected(SelectionEvent event)
			{
				widgetSelected(event);
			}
		});
	}

	protected void handleCompanySelection(LoginShell shell)
	{
		if (shell.supportServerCombo.getSelectionIndex() == -1)
		{
			shell.supportImpersonateCombo.setEnabled(false);
			shell.btnProduction.setSelection(true);
			shell.btnProduction.setEnabled(false);
			shell.btnTest.setEnabled(false);
			return;
		}
		String companyName = shell.supportServerCombo.getItem(shell.supportServerCombo.getSelectionIndex());
		AtomicBoolean hasProduction = new AtomicBoolean();
		AtomicBoolean hasTest = new AtomicBoolean();
		List<Company> matches = companies.stream()
				.filter(c -> c.getName().equals(companyName))
				.peek(c -> {
					switch (c.getEnvironment())
					{
					case Test -> hasTest.set(true);
					case Production -> hasProduction.set(true);
					}
				})
				.toList();
		if (hasProduction.get() && hasTest.get())
		{
			shell.btnProduction.setEnabled(true);
			shell.btnTest.setEnabled(true);
			shell.btnProduction.setSelection(true);
			shell.btnTest.setSelection(false);
			handleEnvironmentSelection(shell);
			return;
		}
		shell.btnProduction.setEnabled(hasProduction.get());
		shell.btnProduction.setSelection(hasProduction.get());
		shell.btnTest.setEnabled(hasTest.get());
		shell.btnTest.setSelection(hasTest.get());
		handleEnvironmentSelection(shell);
	}

	protected Company getSelectedCompany(LoginShell shell)
	{
		Environment environment = shell.btnProduction.getSelection() ? Environment.Production : Environment.Test;
		String companyName = shell.supportServerCombo.getItem(shell.supportServerCombo.getSelectionIndex());
		return companies.stream()
				.filter(c -> c.getName().equals(companyName) && c.getEnvironment() == environment)
				.findFirst()
				.get();
	}

	protected void handleEnvironmentSelection(LoginShell shell)
	{
		Company company = getSelectedCompany(shell);
		usersToImpersonate.clear();
		usersToImpersonate.add(new User(""));
		usersToImpersonate.addAll(users.stream()
				.filter(u -> u.getCompany().equals(company.getName()) && u.getEnvironment() == company.getEnvironment())
				.sorted(Comparator.comparing(this::getUserLabel))
				.toList());
		String[] items = usersToImpersonate.stream()
				.map(this::getUserLabel)
				.toArray(String[]::new);
		shell.supportImpersonateCombo.setItems(items);
		shell.supportImpersonateCombo.select(0);
		shell.supportImpersonateCombo.setEnabled(true);

		validInstances.clear();
		validInstances.addAll(instances.stream()
				.filter(i -> i.namespace().equals(company.getNamespace()))
				.toList());
		if (validInstances.isEmpty())
		{
			shell.namespaceLabel.setText(EMPTY);
			shell.supportInstanceCombo.setItems("<No Instances>");
			shell.supportInstanceCombo.select(0);
			shell.supportInstanceCombo.setEnabled(false);
			shell.okCancelButtons.getOKButton().setEnabled(false);
			return;
		}
		shell.namespaceLabel.setText(company.getNamespace());
		validInstances.add(0, new Instance(company.getNamespace(), EMPTY, true, true));
		Instance newest = validInstances.stream()
				.sorted(Comparator.comparing(Instance::id).reversed())
				.findFirst()
				.get();
		String[] instanceItems = validInstances
				.stream()
				.map(i -> getInstanceLabel(i, newest))
				.toArray(
						String[]::new);
		shell.supportInstanceCombo.setItems(instanceItems);
		shell.supportInstanceCombo.select(0);
		shell.supportInstanceCombo.setEnabled(true);
		shell.okCancelButtons.getOKButton().setEnabled(true);
	}

	private void checkUser(LoginShell shell)
	{
		shell.userEmailResultComposite.setVisible(true);
		shell.userEmailResultCompositeLayout.topControl = shell.userVerifyingEmailComposite;
		shell.loginComposite.layout(true, true);
		shell.userVerifyingEmailImageWidget.play();
		int index = shell.userServerCombo.getSelectionIndex();
		if ((index == -1) && (currentUser.getCompanies().size() == 1))
		{
			index = 0;
		}
		Company company = currentUser.getCompanies().get(index);
		Authenticator.authenticate(OAuthProperties.build(currentUser.getEmail(), company, shell.rememberMeButton.getSelection()))
				.filter(auth -> auth.getIAMProxyToken().length() > 20)
				.doOnComplete(() -> resetCheckUser(shell))
				.flatMap(com.candata.login.oauth.users.oauth2.ValidateToken::validate)
				.onErrorResumeNext(t -> handleError(shell, t))
				.subscribe(response -> async(shell.userEmailResultComposite, () -> validUser(shell, response, currentUser)));
	}

	private Maybe<ValidationResponse> handleError(LoginShell shell, Throwable t)
	{
		logger.log(Level.SEVERE, "error authorizing", t);
		checkUserError(shell, t);
		return Maybe.empty();
	}

	private void resetCheckUser(LoginShell shell)
	{
		async(shell.userEmailResultComposite, () -> {
			shell.leftButtonComposite.setVisible(true);
			shell.okCancelButtons.getOKButton().setEnabled(true);
			shell.userEmailResultComposite.setVisible(true);
			shell.userEmailResultCompositeLayout.topControl = shell.userFailedComposite;
			shell.loginComposite.layout(true, true);
			shell.userFailedIconLabel.setImage(getImage("warn.png"));
			shell.userFailedLabel.setText("Authorization Cancelled");
			shell.userFailedLabel.setToolTipText(
					"Authentication was cancelled. Please try again, or contact support@candata.com.");
		});
	}

	private void checkUserError(LoginShell shell, Throwable t)
	{
		async(shell.userEmailResultComposite, () -> {
			shell.leftButtonComposite.setVisible(true);
			shell.okCancelButtons.getOKButton().setEnabled(true);
			shell.userEmailResultComposite.setVisible(true);
			shell.userEmailResultCompositeLayout.topControl = shell.userFailedComposite;
			shell.loginComposite.layout(true, true);
			shell.userFailedIconLabel.setImage(getImage(FAIL_IMAGE_NAME));
			shell.userFailedLabel.setText("Authorization Error");
			shell.userFailedLabel.setToolTipText(
					"There was an error authenticating. Please try again, or contact support@candata.com.\n"
							+ t.getMessage());
		});
	}

	private void validUser(LoginShell shell, ValidationResponse response, User user)
	{
		shell.leftButtonComposite.setVisible(false);
		shell.okCancelButtons.getOKButton().setEnabled(false);
		shell.userEmailResultComposite.setVisible(true);
		shell.userEmailResultCompositeLayout.topControl = shell.userVerifiedEmailComposite;
		shell.loginComposite.layout(true, true);
		Company company = response.getAuthentication().getCompany();
		LastLogin.save(user.getEmail(), getCompanyLabel(company), shell.rememberMeButton.getSelection());
		shell.userEmailResultComposite.getDisplay().timerExec(500, () -> {
			//start login process
			com.candata.login.oauth.users.oauth2.ValidateToken.decode(response.getAuthentication().getIdToken());
			response.getAuthentication().setCompany(company);
			provision(response.getAuthentication());
		});
	}

	private void startProvision(LoginShell shell, String task, int total)
	{
		if (shell.progressComposite.isDisposed())
		{
			return;
		}
		shell.sl_contentComposite.topControl = shell.progressComposite;
		shell.progressImageWidget.setVisible(total > 0);
		shell.progressLabel.setVisible(total > 0);
		if (total < 1)
		{
			shell.mainProgressBar.beginAnimatedTask();
		}
		else
		{
			shell.mainProgressBar.beginTask(total);
			shell.progressImageWidget.play();
		}
		shell.taskLabel.setText(task);
		shell.contentComposite.layout(true, true);
	}

	private void updateProvision(LoginShell shell, Progress progress)
	{
		if (shell.progressComposite.isDisposed())
		{
			return;
		}
		if (progress.isIndeterminate())
		{
			shell.mainProgressBar.beginAnimatedTask();
		}
		else if (progress.hasValue())
		{
			shell.mainProgressBar.worked(progress.value());
		}
		if (progress.hasDescription())
		{
			shell.taskLabel.setText(progress.description());
		}
		if (progress.hasProgress())
		{
			shell.progressLabel.setText(progress.progress());
		}
	}

	private void checkSupportUser(LoginShell shell)
	{
		if (shell.loginEmailLabel.getText().endsWith(SUPPORT_EMAIL_DOMAIN))
		{
			shell.statusStackLayout.topControl = shell.userValidatingComposite;
			shell.statusErrorComposite.layout();
			shell.userValidatingImageWidget.play();
			Authenticator.authenticate(OAuthSupportProperties.build(shell.loginEmailLabel.getText()))
					.subscribeOn(Schedulers.computation())
					.filter(SupportAuthentications::isValid)
					.flatMap(com.candata.login.oauth.support.oauth2.ValidateToken::getNamespaces)
					.doOnComplete(() -> resetCheckSupportUser(shell))
					.onErrorResumeNext(t -> handleSupportError(shell, t))
					.flatMapCompletable(r -> getAllUsers(shell, r))
					.subscribe();
		}
	}

	private Completable getAllUsers(LoginShell shell, ValidationResponse response)
	{
		return ValidateToken.getAllUsers(response.getAuthentications())
				.map(ValidationResponse::getMessage)
				.map(JSONArray::new)
				.doOnSuccess(allUsers -> async(shell.userEmailResultComposite, () -> validSupportUser(shell, response, allUsers)))
				.ignoreElement();
	}

	private Maybe<ValidationResponse> handleSupportError(LoginShell shell, Throwable t)
	{
		t.printStackTrace();
		async(shell.userEmailResultComposite, () -> {
			shell.sl_contentComposite.topControl = shell.emailComposite;
			shell.emailText.setEnabled(true);
			shell.leftButtonCompositeLayout.topControl = shell.proxyLink;
			shell.okCancelButtons.setAsNext();
			handleOKState(shell);
			shell.emailResultComposite.setVisible(true);
			shell.emailResultCompositeLayout.topControl = shell.invalidEmailComposite;
			shell.invalidEmailLabel.setText("Error authenticating");
			shell.invalidEmailLabel.setToolTipText(t.getMessage());
			shell.contentComposite.getParent().layout(true, true);
		});
		return Maybe.empty();
	}

	private void resetCheckSupportUser(LoginShell shell)
	{
		async(shell.userEmailResultComposite, () -> {
			shell.sl_contentComposite.topControl = shell.emailComposite;
			shell.emailText.setEnabled(true);
			shell.leftButtonCompositeLayout.topControl = shell.proxyLink;
			shell.okCancelButtons.setAsNext();
			handleOKState(shell);
			shell.emailResultComposite.setVisible(false);
			shell.contentComposite.getParent().layout(true, true);
		});
	}

	protected void provision(Authentication authentication)
	{
		System.setProperty("candata.application", authentication.getCompany().getApplication().toString());
		if (authentication.getCompany().getApplication() == Application.Freight)
		{
			System.setProperty("vb6.executable", "something");
		}
		authenticated.onSuccess(authentication);
	}

	private void createProgressComposite(LoginShell shell, Composite progressComposite)
	{
		progressComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		progressComposite.setLayout(new GridLayout(2, false));
		GridData gd_userNameErrorLabel = new GridData(SWT.FILL, SWT.CENTER, true, false);
		gd_userNameErrorLabel.widthHint = 150;

		shell.taskLabel = new Label(progressComposite, SWT.NONE);
		shell.taskLabel.setLayoutData(new GridData(SWT.FILL, SWT.BOTTOM, true, true, 2, 1));
		shell.taskLabel.setText("");
		shell.mainProgressBar = new ProgressIndicator(progressComposite);
		shell.mainProgressBar.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false, 2, 1));
		// mainProgressBar.beginAnimatedTask();
		shell.progressImageWidget = new ImageWidget(progressComposite, getAnimatedImage("waitIcon.gif"));
		GridData gd_progressImageWidget = new GridData(18, 18);
		gd_progressImageWidget.verticalAlignment = SWT.TOP;
		shell.progressImageWidget.setLayoutData(gd_progressImageWidget);
		shell.progressImageWidget.play();
		shell.progressLabel = new Label(progressComposite, SWT.NONE);
		shell.progressLabel.setLayoutData(new GridData(SWT.FILL, SWT.TOP, true, true, 1, 1));
		shell.progressLabel.setText("");
	}

	protected static final String LAST_VALUE = "last.value";

	static final String EMAIL_REGEX = "^(?=.{1,64}@)[A-Za-z0-9_-]+(\\.[A-Za-z0-9_-]+)*@[^-][A-Za-z0-9-]+(\\.[A-Za-z0-9-]+)*(\\.[A-Za-z]{2,})$";
	static final Pattern EMAIL_PATTERN = Pattern.compile(EMAIL_REGEX);
	private Map<String, Image> images = new HashMap<>();

	private SupportAuthentications supportAuthentications;

	private List<Company> companies;
	private List<Instance> instances;
	private List<Instance> validInstances = new ArrayList<>();
	private List<User> users = new ArrayList<>();
	private List<User> usersToImpersonate = new ArrayList<>();
	private static final String INVALID_EMAIL = "Invalid Email";
	private static final String AUTHENTICATION_FAILED = "Authentication Failed";
	private static final String ERROR_VALIDATING_EMAIL = "Error validating email";
	private static final String FAIL_IMAGE_NAME = "error.png";
	private static final String BUILD_PREFIX = "build-";
	private User currentUser;
	ProgressMonitor progressMonitor;
}
