package com.candata.login.zoo.osgi;

import static com.candata.login.utils.StringUtils.COMMA;
import static java.util.stream.Collectors.joining;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import java.util.logging.FileHandler;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Stream;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.Constants;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.framework.launch.Framework;
import org.osgi.framework.launch.FrameworkFactory;
import org.osgi.framework.wiring.BundleCapability;
import org.osgi.framework.wiring.BundleWiring;
import org.osgi.service.log.LogReaderService;
import org.osgi.util.tracker.ServiceTracker;

import com.candata.login.oauth.users.Company;
import com.candata.login.utils.StringUtils;
import com.candata.login.zoo.os.SWT;
import com.candata.login.zoo.osgi.tools.LoggerImpl;

import io.reactivex.rxjava3.core.Single;

public class OSGiStarter
{

	public static Single<Configuration> start(Company company)
	{
		return Single.fromCallable(() -> new OSGiStarter().init(company))
				.timeout(20, TimeUnit.SECONDS);
	}

	protected Configuration init(Company company) throws Exception
	{
		try (WorkingFolder workingFolder = new WorkingFolder(company))
		{
			File workDir = workingFolder.getDirectory();
			logger = createLogger(workDir);
			logger.log(Level.INFO, "Working Directory: " + workDir.getAbsolutePath());
			logger.log(Level.INFO, "OS: " + System.getProperty("os.name") + " " + System.getProperty("os.arch"));
			Properties properties = System.getProperties();
			StringBuilder output = new StringBuilder("Properties:\n");
			properties.forEach((k, v) -> output.append("  " + k + ": " + v + "\n"));
			logger.log(Level.INFO, output.toString());
			Map<String, String> osgiProp = getProperties(workDir);
			ClassLoader loader = getClass().getClassLoader();
			List<String> implementations = getMetaInfServices(loader, FrameworkFactory.class.getName());
			if (implementations.size() == 0)
			{
				error("Found no fw implementation");
				System.exit(1);
			}
			else if (implementations.size() > 1)
			{
				error("Found more than one framework implementations: " + implementations);
				System.exit(1);
			}

			String implementation = implementations.get(0);
			Class<?> clazz = loader.loadClass(implementation);
			FrameworkFactory factory = (FrameworkFactory) clazz.newInstance();
			Framework systemBundle = factory.newFramework(osgiProp);
			systemBundle.init();

			//			boolean restart = false;
			//			do
			//			{
			//				{
			registerLogger(systemBundle.getBundleContext());
			displayProperties(systemBundle);
			start(systemBundle, workingFolder);
			logger.log(Level.INFO, "OSGi Started.  Bundles: " + systemBundle.getBundleContext().getBundles().length);
			//					logger.log(Level.INFO, "clear: " + System.getProperty(CLEAR_CACHE));
			//					restart = (System.getProperty(CLEAR_CACHE) != null);
			return new Configuration(systemBundle, workingFolder);
			//				}
			//				if (restart)
			//				{
			//					if (frameworkStoppingListeners != null)
			//					{
			//						frameworkStoppingListeners.close();
			//					}
			//					System.gc();
			//					//					restart = doClear(systemBundle, workingFolder);
			//					//					ctx = null;
			//					//					logger.log(Level.INFO, "\n\n\n" + showThreads());
			//					//					logger.log(Level.INFO, "\n\n\nRestarting");
			//				}
			//			}
			//			while (restart);
			//			systemBundle.stop();
			//			logger.log(Level.INFO, "Application stopping");
			//			System.exit(0);
		}
		catch (Exception e)
		{
			e.printStackTrace();
			logger.log(Level.SEVERE, "Application exiting due to error", e);
			System.exit(1);
			throw e;
		}
	}

	private List<String> getMetaInfServices(ClassLoader loader, String factory)
			throws IOException
	{
		return Collections.list(loader.getResources("META-INF/services/" + factory))
				.stream()
				.map(this::getFactories)
				.flatMap(List::stream)
				.toList();
	}

	private List<String> getFactories(URL url)
	{
		try (BufferedReader reader = new BufferedReader(new InputStreamReader(url.openStream())))
		{
			return reader
					.lines()
					.filter(line -> !line.startsWith(HASH) && !line.isBlank())
					.toList();
		}
		catch (IOException e)
		{
			throw new RuntimeException(e);
		}
	}

	protected Map<String, String> getProperties(File workingDir) throws Exception
	{
		Map<String, String> osgiProp = new HashMap<>();
		osgiProp.put("eclipse.ignoreApp", "true");
		osgiProp.put("osgi.startLevel", APP_START_LEVEL.toString());
		osgiProp.put("osgi.compatibility.bootdelegation", "true");
		if (workingDir == null)
		{
			throw new Exception("Unable to find a working folder: " + workingDir);
		}
		osgiProp.put("osgi.configuration.area", workingDir.getAbsolutePath());
		osgiProp.put("osgi.framework.activeThreadType", "none");
		String debugFile = System.getProperty("osgi.debug");
		if (debugFile != null)
		{
			osgiProp.put("osgi.debug", debugFile);
		}

		String dsDebug = System.getProperty("ds.loglevel");
		if (dsDebug != null)
		{
			osgiProp.put("ds.loglevel", dsDebug);
		}
		String exported = Stream.concat(Stream.of(PACKAGES_TO_EXPORT), SWT.getPackagesToExport())
				.collect(joining(COMMA))
				.replace("\n", StringUtils.EMPTY);
		osgiProp.put(Constants.FRAMEWORK_SYSTEMPACKAGES_EXTRA, exported);
		return osgiProp;
	}

	protected void start(Framework systemBundle, WorkingFolder workingFolder) throws Exception
	{
		this.ctx = systemBundle.getBundleContext();
		frameworkStoppingListeners = new ServiceTracker<>(ctx,
				ctx.createFilter("(&(objectclass=java.lang.Runnable)(Runtime.ShutdownHook=*))"), null);
		frameworkStoppingListeners.open(true);
		hookShutdownListener(systemBundle, workingFolder);
		//		provisionAndStart(systemBundle, workingFolder, args);
	}

	protected void hookShutdownListener(Framework systemBundle, WorkingFolder workingFolder)
	{
		if (System.getProperty(SHUTDOWNHOOK_INSTALLED) != null)
		{
			return;
		}
		System.setProperty(SHUTDOWNHOOK_INSTALLED, "true");
		Runtime.getRuntime().addShutdownHook(new Thread(() -> {
			try
			{
				error("Calling the Framework Stopping listeners");
				Object shutdownHooks[] = frameworkStoppingListeners.getServices();
				if (shutdownHooks != null)
				{
					for (Object shutdown : shutdownHooks)
					{
						error("Calling " + shutdown.getClass());
						((Runnable) shutdown).run();
					}
				}
				frameworkStoppingListeners.close();

				error("Shutting down the framework");

				systemBundle.stop();
				error("Shutting down sent waiting for bundles");

				workingFolder.close();
				error("Shutting down");
				Runtime.getRuntime().halt(0);
			}
			catch (Throwable e)
			{
				e.printStackTrace();
			}
		}));
	}

	protected static void displayProperties(Bundle bundle)
	{
		BundleWiring bundleWiring = bundle.adapt(BundleWiring.class);
		List<BundleCapability> capabilities = bundleWiring.getCapabilities("osgi.native");
		StringBuilder properties = new StringBuilder();
		properties.append("osgi.native:\n");
		for (BundleCapability capability : capabilities)
		{
			for (Map.Entry<String, Object> attr : capability.getAttributes().entrySet())
			{
				properties.append("   " + attr.getKey() + " " + attr.getValue() + "\n");
			}
		}
		error(properties.toString());
	}

	public static void error(String message)
	{
		System.err.println(message);
		if (logger != null)
		{
			logger.info(message);
		}
	}

	protected void registerLogger(BundleContext ctx) throws InvalidSyntaxException
	{
		@SuppressWarnings("unchecked")
		ServiceReference<LogReaderService>[] refs = (ServiceReference<LogReaderService>[]) ctx
				.getAllServiceReferences(LogReaderService.class.getName(), null);
		if (refs != null)
		{
			ctx.getService(refs[0]).addLogListener(new LoggerImpl());
		}
	}

	protected static Logger createLogger(File workDir)
	{
		Logger logger = Logger.getLogger("Candata");
		logger.setLevel(java.util.logging.Level.ALL);
		try
		{
			// This block configure the logger with handler and formatter
			File logFileFolder = new File(workDir, "/logs");
			tidyOldLogs(logFileFolder);
			logFileFolder.mkdirs();
			String filename = DateTimeFormatter.ofPattern("yyyyMMddHHmmss").format(LocalDateTime.now()) + ".log";
			File logFile = new File(logFileFolder, filename);
			FileHandler fh = new FileHandler(logFile.getAbsolutePath());
			fh.setFormatter(new CandataFormatter());
			logger.addHandler(fh);
		}
		catch (Exception e)
		{
			logger.log(Level.SEVERE, "Error adding log file", e);
			e.printStackTrace();
		}
		return logger;
	}

	protected static void tidyOldLogs(File logFolder)
	{
		try
		{
			if (logFolder.exists())
			{
				for (File file : logFolder.listFiles())
				{
					if (Instant.ofEpochMilli(file.lastModified()).isBefore(LocalDateTime.now().minusMonths(1).toInstant(ZoneOffset.UTC)))
					{
						file.delete();
					}
				}
			}
		}
		catch (Throwable t)
		{
			logger.log(Level.SEVERE, "problem tidying old logs", t);
		}
	}

	static Logger logger;
	protected BundleContext ctx;

	protected ServiceTracker<Runnable, Runnable> frameworkStoppingListeners;

	protected static final Integer APP_START_LEVEL = 2;
	protected static final int START_LEVEL = 6;
	protected static final String SHUTDOWNHOOK_INSTALLED = "shutdownHook.installed";

	static final String HASH = "#";
	static final String PACKAGES_TO_EXPORT[] = new String[] {};
}
