package com.candata.login.zoo.users;

import static com.candata.login.oauth.oauth2.Constants.TEST_REALM;
import static com.candata.login.utils.StringUtils.EMPTY;

import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Stream;

import org.json.JSONArray;
import org.json.JSONObject;

import com.candata.login.oauth.beans.Environment;
import com.candata.login.oauth.oauth2.Constants;
import com.candata.login.oauth.users.Application;
import com.candata.login.oauth.users.Company;
import com.candata.login.oauth.users.User;

public class UserParser
{
	public static User parse(JSONObject data)
	{
		User user = new User(data.getString(EMAIL_KEY));
		user.setFirstName(data.optString(UserParser.FIRSTNAME_KEY));
		user.setLastName(data.optString(UserParser.LASTNAME_KEY));
		user.setCompany(data.optString(UserParser.COMPANY_NAME_KEY).replace(Constants.TEST_REALM, EMPTY));
		user.setNamespace(data.optString(NAMESPACE_KEY));
		user.setApplication(Application.get(data.optString(APPLICATION_KEY)));
		user.setEnvironment(data.optString(COMPANY_NAME_KEY).endsWith(Constants.TEST_REALM) ? Environment.Test : Environment.Production);
		if (data.has(COMPANIES_KEY))
		{
			parseCompanies(data.getJSONArray(COMPANIES_KEY))
					.forEach(user::add);
		}
		return user;
	}

	protected static Stream<Company> parseCompanies(JSONArray companiesJson)
	{
		List<Company> companies = new ArrayList<>();
		companiesJson.forEach(c -> companies.add(parseCompany((JSONObject) c)));
		return companies.stream();
	}

	protected static Company parseCompany(JSONObject companyJson)
	{
		Company company = new Company();
		company.setApplication(Application.get(companyJson.getString(APPLICATION_KEY)));
		company.setName(companyJson.getString(COMPANY_NAME_KEY));
		company.setClientSecret(companyJson.getString(CLIENT_SECRET_KEY));
		company.setNamespace(companyJson.getString(NAMESPACE_KEY));
		company.setEnvironment(Environment.Production);
		if (company.getName().endsWith(TEST_REALM))
		{
			company.setEnvironment(Environment.Test);
			company.setName(company.getName().replace(TEST_REALM, EMPTY));
		}
		return company;
	}

	protected static OffsetDateTime toOffset(long value)
	{
		return (value == 0) ? null : OffsetDateTime.ofInstant(Instant.ofEpochMilli(value), ZoneOffset.UTC);
	}

	protected static final String ENVIRONMENT_KEY = "environment";
	protected static final String COMPANIES_KEY = "companies";
	protected static final String EMAIL_KEY = "email";
	protected static final String FIRSTNAME_KEY = "firstName";
	protected static final String LASTNAME_KEY = "lastName";
	protected static final String COMPANY_NAME_KEY = "company";
	protected static final String CLIENT_SECRET_KEY = "secret";
	protected static final String NAMESPACE_KEY = "namespace";
	protected static final String APPLICATION_KEY = "application";

	/**
	{
	"email": "jamiedougal@gmail.com",
	"companies": [
	    {
	        "email": "jamiedougal@gmail.com",
	        "company": "candatatest",
	        "client": "EOSMzWO7EzGkzCeQ5M5BxMA4hsMff27Q",
	        "namespace": "test"
	    },
	    {
	        "email": "jamiedougal@gmail.com",
	        "company": "mckaytest",
	        "client": "CR106dUjVFyUOFuVsKeoi52ccxgiuRMa",
	        "namespace": "playground"
	    }
	]
	}
	 */

}
