package com.candata.login.zoo.users;

import static com.candata.login.zoo.users.UserParser.COMPANIES_KEY;
import static com.candata.login.zoo.users.UserParser.EMAIL_KEY;
import static java.text.MessageFormat.format;

import java.net.URI;
import java.net.http.HttpRequest.BodyPublisher;
import java.net.http.HttpRequest.BodyPublishers;
import java.net.http.HttpRequest.Builder;
import java.net.http.HttpResponse.BodyHandlers;

import org.json.JSONArray;
import org.json.JSONObject;

import com.candata.login.oauth.beans.Environment;
import com.candata.login.oauth.users.User;
import com.candata.login.utils.HttpUtils;
import com.candata.login.utils.HttpUtils.ContentType;
import com.candata.login.utils.Version;

import io.reactivex.rxjava3.core.Maybe;
import io.reactivex.rxjava3.core.Single;
import io.reactivex.rxjava3.schedulers.Schedulers;

public class UserValidation
{
	public static Maybe<User> isValid(String email)
	{
		return Single.fromCallable(Key::new)
				.flatMapMaybe(key -> doRequest(email, key))
				.subscribeOn(Schedulers.computation())
				.map(UserParser::parse);
	}

	protected static Maybe<JSONObject> doRequest(String email, Key key)
	{
		return HttpUtils.buildPost(getURI(), ContentType.JSON, buildBody(email, key))
				.map(Builder::build)
				.flatMapMaybe(r -> HttpUtils.doRequest(r, BodyHandlers.ofByteArray()))
				.map(d -> decrypt(d, key))
				.filter(j -> j.length() > 0)
				.map(j -> wrap(email, j));
	}

	protected static JSONArray decrypt(byte[] data, Key key) throws Exception
	{
		return new JSONArray(Key.decrypt(data, key));
	}

	protected static BodyPublisher buildBody(String email, Key key)
	{
		JSONObject data = new JSONObject();
		data.put(EMAIL_KEY, email);
		data.put(VERSION_KEY, Version.VERSION);
		data.put(SHARED_KEY, key.getSharedKey());
		return BodyPublishers.ofString(data.toString());
	}

	protected static JSONObject wrap(String email, JSONArray companies)
	{
		JSONObject result = new JSONObject();
		result.put(EMAIL_KEY, email);
		result.put(COMPANIES_KEY, companies);
		return result;
	}

	protected static URI getURI()
	{
		return URI.create(USER_VALIDATION_BASE_URL);
	}

	private static final Environment ENVIRONMENT = Environment.get();

	private static final String USER_VALIDATION_BASE_URL = format("https://api-proxy{0}.candata.com/login/userInfo", ENVIRONMENT.getURLInsert());
	private static final String VERSION_KEY = "version";
	private static final String SHARED_KEY = "shared";
}
