package com.candata.login.zoo.widgets;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;

public class ImageWidget extends Canvas implements Runnable
{
	Color shellBackground;

	public ImageWidget(Composite parent, int style)
	{
		super(parent, style);
	}

	public ImageWidget(Composite parent, final ImageData[] imageData)
	{
		super(parent, SWT.TRANSPARENT | SWT.DOUBLE_BUFFERED | SWT.NO_BACKGROUND);
		imageDataArray = imageData;
		if (imageData == null)
		{
			return;
		}
		if (imageDataArray.length > 1)
		{
			currentImageData = imageDataArray[0];
		}

		// listen for move/resize events to adjust location of painting
		addControlListener(new ControlListener() {
			@Override
			public void controlMoved(ControlEvent event)
			{
				if (imageDataArray == null || imageDataArray.length == 0)
				{
					return;
				}
				ImageWidget me = ImageWidget.this;
				x = me.getBounds().x + (me.getBounds().width - imageDataArray[0].width) / 2;
				y = me.getBounds().y + (me.getBounds().height - imageDataArray[0].height) / 2;
			}

			@Override
			public void controlResized(ControlEvent event)
			{
				if (imageDataArray == null || imageDataArray.length == 0)
				{
					return;
				}
				ImageWidget me = ImageWidget.this;
				x = me.getBounds().x + (me.getBounds().width - imageDataArray[0].width) / 2;
				y = me.getBounds().y + (me.getBounds().height - imageDataArray[0].height) / 2;
			}

		});
		shellBackground = getShell().getBackground();
		addPaintListener(event -> {
			if (isStopped)
			{
				event.gc.setBackground(shellBackground);
				return;
			}
			if (imageDataArray == null || imageDataArray.length == 0)
			{
				return;
			}
			int increment = (getData(REDRAW_KEY) != null) ? 1 : 0;
			setData(REDRAW_KEY, null);
			event.gc.setBackground(shellBackground);
			imageDataIndex = (imageDataIndex + increment) % imageDataArray.length;
			if (singleLoop && singleLoopDone)
			{
				imageDataIndex = imageDataArray.length - 1;
			}
			currentImageData = imageDataArray[imageDataIndex];
			Image currentImage = new Image(event.display, currentImageData);
			int destWidth = ImageWidget.this.getBounds().width;
			int destHeight = ImageWidget.this.getBounds().height;

			event.gc.drawImage(currentImage, 0, 0,
					currentImageData.width, currentImageData.height,
					0 + horizontalMargin, 0 + verticalMargin,
					destWidth, destHeight);
			currentImage.dispose();
		});
		getShell().addDisposeListener(event -> dispose());
	}

	public void setSingleLoop()
	{
		this.singleLoop = true;
	}

	@Override
	public void dispose()
	{
		if (shellBackground != null && !shellBackground.isDisposed())
		{
			shellBackground.dispose();
			shellBackground = null;
		}
	}

	public int getImageWidth()
	{
		if (imageDataArray == null || imageDataArray.length == 0)
		{
			return 0;
		}
		return imageDataArray[0].width;
	}

	public int getImageHeight()
	{
		if (imageDataArray == null || imageDataArray.length == 0)
		{
			return 0;
		}
		return imageDataArray[0].height;
	}

	public void play()
	{
		setStopped(false);
	}

	public void stop()
	{
		setStopped(true);
	}

	public boolean isStopped()
	{
		return isStopped;
	}

	private void setStopped(boolean value)
	{
		isStopped = value;
		if (!isStopped)
		{
			getDisplay().asyncExec(this);
		}
	}

	public void loop()
	{
		singleLoopDone = false;
		imageDataIndex = 0;
		redraw();
	}

	public void setHorizontalMargin(int horizontalMargin)
	{
		this.horizontalMargin = horizontalMargin;
	}

	public void setVerticalMargin(int verticalMargin)
	{
		this.verticalMargin = verticalMargin;
	}

	@Override
	public void run()
	{
		if (isDisposed() || !isVisible())
		{
			return;
		}
		setData(REDRAW_KEY, "true");
		redraw();
		if (singleLoop && imageDataIndex + 1 == imageDataArray.length)
		{
			singleLoopDone = true;
		}
		if (isStopped())
		{
			return;
		}

		int sleepTime = currentImageData.delayTime * 10;
		if (sleepTime < 20)
		{
			sleepTime += 30;
		}
		if (sleepTime < 30)
		{
			sleepTime += 10;
		}
		getDisplay().timerExec(sleepTime, this);
	}

	private static final String REDRAW_KEY = "redraw.issued";
	private ImageData currentImageData;
	private int imageDataIndex;
	private ImageData[] imageDataArray;
	final boolean useGIFBackground = false;
	private boolean isStopped = false;
	private int x;
	private int y;
	private boolean singleLoop = false;
	private boolean singleLoopDone = false;
	private int verticalMargin = 0;
	private int horizontalMargin = 0;
}
