package com.candata.login.zoo;

import static com.candata.login.zoo.utils.DisplayThread.async;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.stream.Stream;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.DeviceData;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.launch.Framework;

import com.candata.login.oauth.TokenRefreshServiceImpl;
import com.candata.login.oauth.beans.Authentication;
import com.candata.login.utils.Platform;
import com.candata.login.zoo.exceptions.UnknownUserException;
import com.candata.login.zoo.osgi.Configuration;
import com.candata.login.zoo.osgi.OSGiStarter;
import com.candata.login.zoo.provision.Provisioner;
import com.candata.login.zoo.provision.impl.ProgressMonitorUpdaterImpl;
import com.candata.login.zoo.provision.interfaces.ProgressMonitor;
import com.candata.login.zoo.utils.WindowsUtils;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Maybe;
import io.reactivex.rxjava3.schedulers.Schedulers;
import io.reactivex.rxjava3.subjects.MaybeSubject;

public class LoginStarter
{

	public void start()
	{
		try
		{
			Platform.initializeToolkit();

			DeviceData data = new DeviceData();
			data.tracking = true;
			Display display = new Display(data);
			WindowsUtils.configureZoom(display);
			WindowsUtils.hookDarkModeListeners(display);

			Login login = new Login(display);
			Shell loginShell = login.getLoginShell();
			ProgressMonitor monitor = login.getProgressMonitor();
			login.setAuthenticationCallback(buildAuthCallback(display, loginShell, new Launcher(), monitor));

			AtomicBoolean doLoop = new AtomicBoolean(!display.isDisposed());
			while (doLoop.get())
			{
				if (!display.readAndDispatch())
				{
					display.sleep();
				}
				if (keepAlive(display))
				{
					if (display.getShells().length == 2 || hasApplicationShell(display))
					{
						monitor.complete();
					}
				}
				doLoop.set(keepAlive(display));
			}
			tidy();
		}
		catch (Throwable e)
		{
			e.printStackTrace();
			logger.log(Level.SEVERE, "error on login screen", e);
			throw new RuntimeException(e);
		}
	}

	protected boolean hasApplicationShell(Display display)
	{
		//		setData(DialogService.APPLICATION_SHELL, "true");
		return Stream.of(display.getShells())
				.anyMatch(s -> !s.isDisposed() && s.getData(APPLICATION_SHELL) != null);
	}

	protected boolean keepAlive(Display display)
	{
		return !display.isDisposed() && System.getProperty(CANDATA_SHUTDOWN_KEY) == null && System.getProperty(LOGIN_CANCELLED_KEY) == null;
	}

	protected void tidy()
	{
		if (refreshServiceRegistration != null)
		{
			try
			{
				refreshServiceRegistration.unregister();
			}
			catch (IllegalStateException e)
			{
			}
		}
		if (progressUpdaterRegistration != null)
		{
			try
			{
				progressUpdaterRegistration.unregister();
			}
			catch (IllegalStateException e)
			{
			}
		}
		if (configuration != null)
		{
			try
			{
				logger.log(Level.INFO, "Shutting down");
				configuration.systemBundle().stop();
			}
			catch (BundleException e)
			{
				logger.log(Level.SEVERE, "error shutting down", e);
			}
			logger.log(Level.INFO, "Unlocking working folder " + configuration.workingFolder().getDirectory().getAbsolutePath());
			configuration.workingFolder().close();
		}
	}

	protected MaybeSubject<Authentication> buildAuthCallback(Display display, Shell shell, Launcher launcher, ProgressMonitor monitor)
	{
		MaybeSubject<Authentication> authCallback = MaybeSubject.create();
		authCallback
				.subscribeOn(Schedulers.computation())
				.observeOn(Schedulers.computation())
				.flatMapCompletable(auth -> initOSGiAndProvision(display, shell, launcher, monitor, auth))
				.doOnError(e -> logger.log(Level.SEVERE, "failed to start the OSGi container", e))
				.subscribe();
		return authCallback;
	}

	protected Completable initOSGiAndProvision(Display display, Shell loginShell, Launcher launcher, ProgressMonitor monitor, Authentication auth)
	{
		return OSGiStarter.start(auth.getCompany())
				.doOnSuccess(c -> configuration = c)
				.map(Configuration::systemBundle)
				.flatMapCompletable(sb -> startOSGiAndProvision(display, loginShell, launcher, monitor, auth, sb));
	}

	protected Completable startOSGiAndProvision(Display display, Shell loginShell, Launcher launcher, ProgressMonitor monitor, Authentication auth, Framework systemBundle)
	{
		return Maybe.fromCallable(systemBundle::getBundleContext)
				.doOnSuccess(ctx -> registerServices(ctx, monitor))
				.flatMapCompletable(ctx -> new Provisioner().provision(ctx, monitor, auth)
						.andThen(Completable.fromAction(systemBundle::start))
						.andThen(Completable.fromAction(() -> doRun(display, launcher, monitor, ctx))))
				.doOnError(t -> {
					if (t.getCause() instanceof UnknownUserException)
					{
						async(loginShell, () -> showUnknownUser(loginShell, ((UnknownUserException) t.getCause()).getEmail(), monitor));
						return;
					}
					async(loginShell, () -> showFail(loginShell, monitor));
				});
	}

	protected void registerServices(BundleContext ctx, ProgressMonitor monitor)
	{
		refreshServiceRegistration = TokenRefreshServiceImpl.register(ctx);
		progressUpdaterRegistration = ProgressMonitorUpdaterImpl.register(ctx, monitor);
	}

	protected void doRun(Display display, Launcher launcher, ProgressMonitor monitor, BundleContext ctx)
	{
		display.asyncExec(() -> {
			if (!launcher.run(ctx, monitor))
			{
				System.setProperty(LOGIN_CANCELLED_KEY, "true");
			}
		});
	}

	protected void showUnknownUser(Shell shell, String email, ProgressMonitor monitor)
	{
		System.setProperty(LOGIN_CANCELLED_KEY, "true");
		if (shell.isDisposed())
		{
			return;
		}
		MessageBox messageBox = new MessageBox(shell, SWT.ICON_WARNING | SWT.OK);
		messageBox.setText("Unknown User");
		messageBox.setMessage("The application doesn't recognize the email address " + email + ".  Please contact support@candata.com.");
		messageBox.open();
		monitor.complete();
	}

	protected void showFail(Shell shell, ProgressMonitor monitor)
	{
		System.setProperty(LOGIN_CANCELLED_KEY, "true");
		if (shell.isDisposed())
		{
			return;
		}
		MessageBox messageBox = new MessageBox(shell, SWT.ICON_WARNING | SWT.OK);
		messageBox.setText("Error");
		messageBox.setMessage("The application failed to start.  Please try again or contact support@candata.com.");
		messageBox.open();
		monitor.complete();
	}

	BundleContext ctx;
	ServiceRegistration<Consumer> loginRegistration;
	ServiceRegistration<Consumer> progressUpdaterRegistration;
	ServiceRegistration<Function> refreshServiceRegistration;
	Configuration configuration;
	protected static java.util.logging.Logger logger = java.util.logging.Logger.getLogger("Candata");
	protected static final String LOGIN_CANCELLED_KEY = "loginCancelled";
	protected static final String CANDATA_SHUTDOWN_KEY = "candata.shutdown";
	protected static String APPLICATION_SHELL = "application.shell";
}
