package com.candata.login.zoo.provision;

import static java.math.RoundingMode.HALF_UP;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.util.Objects;
import java.util.function.Consumer;

public class MonitoredInputStream extends InputStream
{
	public MonitoredInputStream(InputStream inputStream, Consumer<Integer> progressUpdater)
	{
		this.progressUpdater = progressUpdater;
		this.inputStream = inputStream;
	}

	@Override
	public int read() throws IOException
	{
		int length = inputStream.read();
		if (length != -1)
		{
			downloadedLength++;
			updateIfNecessary();
		}
		return length;

	}

	@Override
	public int read(byte b[], int off, int len) throws IOException
	{
		int length = inputStream.read(b, off, len);
		if (length != -1)
		{
			downloadedLength += length;
			updateIfNecessary();
		}
		return length;
	}

	protected void updateIfNecessary() throws IOException
	{
		if (progressUpdater == null)
		{
			return;
		}
		if (downloadedLength - lastUpdated > 100000)
		{
			progressUpdater.accept(downloadedLength);
			lastUpdated = downloadedLength;
		}
	}

	protected String getSize(int length)
	{
		if (length < 1000000)
		{
			return new BigDecimal(length).divide(KB).setScale(0, HALF_UP).toString() + " KB";
		}
		return new BigDecimal(length).divide(MB).setScale(1, HALF_UP).toString() + " MB";
	}

	@Override
	public long skip(long n) throws IOException
	{
		downloadedLength += n;
		return inputStream.skip(n);
	}

	@Override
	public int available() throws IOException
	{
		return inputStream.available();
	}

	@Override
	public void close() throws IOException
	{
		inputStream.close();
	}

	@Override
	public void mark(int readlimit)
	{
		inputStream.mark(readlimit);
	}

	@Override
	public void reset() throws IOException
	{
		downloadedLength = 0;
		inputStream.reset();
	}

	@Override
	public boolean markSupported()
	{
		return inputStream.markSupported();
	}

	public long transferTo(OutputStream output, int length) throws IOException
	{
		Objects.requireNonNull(output, "output");
		long transferred = 0;
		int read;
		int remaining = length;
		while (remaining > 0 && (read = this.read(buffer, 0, Math.min(remaining, DEFAULT_BUFFER_SIZE))) >= 0)
		{
			remaining -= read;
			output.write(buffer, 0, read);
			transferred += read;
		}
		return transferred;
	}

	Consumer<Integer> progressUpdater;
	InputStream inputStream;
	int lastUpdated = 0;
	int downloadedLength = 0;
	protected BigDecimal KB = new BigDecimal(1000);
	protected BigDecimal MB = new BigDecimal(1000000);
	private static final int DEFAULT_BUFFER_SIZE = 8192;
	byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
}