package com.candata.login.zoo.provision;

import static com.candata.login.utils.HttpUtils.AUTHORIZATION_HEADER;
import static com.candata.login.utils.StringUtils.EMPTY;
import static com.candata.login.zoo.utils.Properties.RX_ID_PROPERTY;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpRequest.BodyPublishers;
import java.net.http.HttpRequest.Builder;
import java.net.http.HttpResponse.BodyHandlers;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.text.MessageFormat;
import java.util.Comparator;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import java.util.zip.GZIPInputStream;

import org.eclipse.swt.widgets.Display;
import org.json.JSONArray;
import org.json.JSONObject;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.osgi.framework.ServiceRegistration;
import org.osgi.framework.startlevel.BundleStartLevel;
import org.osgi.framework.wiring.BundleRevision;
import org.osgi.framework.wiring.FrameworkWiring;

import com.candata.login.oauth.beans.Authentication;
import com.candata.login.oauth.beans.Environment;
import com.candata.login.oauth.oauth2.SupportAuthenticator;
import com.candata.login.oauth.support.oauth2.beans.OAuthSupportProperties;
import com.candata.login.utils.FolderUtils;
import com.candata.login.utils.HttpUtils;
import com.candata.login.utils.StringUtils;
import com.candata.login.utils.Version;
import com.candata.login.zoo.exceptions.UnknownUserException;
import com.candata.login.zoo.provision.beans.ProvisionBundle;
import com.candata.login.zoo.provision.interfaces.Progress;
import com.candata.login.zoo.provision.interfaces.ProgressMonitor;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Observable;
import io.reactivex.rxjava3.schedulers.Schedulers;
import io.reactivex.rxjava3.subjects.CompletableSubject;

public class Provisioner
{
	public static void main(String[] args)
	{
		Display display = Display.getDefault();
		AtomicBoolean done = new AtomicBoolean();
		System.setProperty("candata.workingDir", "/tmp/candatatest");
		System.setProperty(FolderUtils.CANDATA_DIR_PROPERTY, "/tmp/repository");
		SupportAuthenticator.authenticate(OAuthSupportProperties.build("jamie.dougal@candata.com"))
				.subscribeOn(Schedulers.computation())
				.subscribe(auths -> {
					Provisioner service = new Provisioner();
					service.provision(null, getProgressMonitor(), auths.getTest()).blockingAwait();
					done.set(true);
				});
		while (!done.get() && !display.isDisposed())
		{
			if (!display.readAndDispatch())
			{
				display.sleep();
			}
		}
	}

	private static ProgressMonitor getProgressMonitor()
	{
		return new ProgressMonitor() {
			@Override
			public void update(Progress progress)
			{

			}

			@Override
			public void configure(String task, int total)
			{

			}

			@Override
			public void complete()
			{

			}
		};
	}

	public Completable provision(BundleContext ctx, ProgressMonitor monitor, Authentication authentication)
	{
		this.ctx = ctx;
		monitor.configure("Checking for updates");
		AtomicInteger bundleCount = new AtomicInteger();
		Observable<Bundle> bundleObs = ctx == null ? Observable.empty() : Observable.fromArray(ctx.getBundles());
		return bundleObs
				.filter(this::isUIBundle)
				.doOnNext(b -> bundleCount.incrementAndGet())
				.map(this::toJSON)
				.reduce(new JSONArray(), JSONArray::put)
				.map(this::addInfo)
				.doOnSuccess(j -> logger.log(Level.INFO, "current bundle count: " + bundleCount.get()))
				.flatMapCompletable(data -> doProvision(monitor, authentication, data));
	}

	protected boolean isUIBundle(Bundle bundle)
	{
		int startLevel = bundle.adapt(BundleStartLevel.class).getStartLevel();
		return startLevel > START_LEVEL;
	}

	protected Completable doProvision(ProgressMonitor monitor, Authentication authentication, JSONObject data)
	{
		CompletableSubject result = CompletableSubject.create();
		logger.log(Level.INFO, "provision: " + data);
		HttpRequest request = buildRequest(authentication)
				.POST(BodyPublishers.ofString(data.toString()))
				.build();
		HttpClient.newHttpClient().sendAsync(request, BodyHandlers.ofInputStream())
				.thenApplyAsync(response -> {
					int status = response.statusCode();
					if (status == HttpURLConnection.HTTP_UNAUTHORIZED)
					{
						throw new UnknownUserException(authentication.getEmail());
					}
					if (status != HttpURLConnection.HTTP_OK)
					{
						try (InputStream errorData = response.body())
						{
							String headers = "URL: " + request.uri() + "\nRequest Headers\n" + HttpUtils.headers(request) + "\nResponse Headers\n"
									+ HttpUtils.headers(response);
							System.err.println(
									"Provision request failed.\n(" + response.statusCode() + ") " + new String(errorData.readAllBytes()) + "\n"
											+ headers);
							result.onError(new RuntimeException(
									"Provision request failed.\n(" + response.statusCode() + ") " + new String(errorData.readAllBytes()) + "\n"
											+ headers));
						}
						catch (Exception e)
						{
							result.onError(e);
						}
						return null;
					}
					try (InputStream responseData = response.body())
					{
						String jsonBody = new String(responseData.readNBytes(getInfoLength(responseData)));
						if (jsonBody.length() == 0)
						{
							result.onError(new RuntimeException("Provision response was empty."));
							return null;
						}
						JSONObject info = new JSONObject(jsonBody);
						try (MonitoredInputStream monitored = new MonitoredInputStream(responseData, monitor.progressUpdater()))
						{
							processResponse(authentication, monitored, info, monitor);
						}
						result.onComplete();
						return null;
					}
					catch (Exception e)
					{
						e.printStackTrace();
						result.onError(e);
					}
					return null;
				})
				.join();
		return result;
	}

	protected Builder buildRequest(Authentication authentication)
	{
		Builder builder = HttpRequest.newBuilder()
				.uri(getURI(authentication))
				.header("Content-Type", "application/json")
				.header("X-CG-Company", authentication.getCompany().getName())
				.header("X-CG-Realm", authentication.getCompany().getRealm())
				.header("X-CG-ID", authentication.getIdToken())
				.header("X-CG-Namespace", authentication.getNamespace())
				.header("X-OS-Arch", System.getProperty("os.arch"))
				.header("X-OS-Name", System.getProperty("os.name"))
				.header(AUTHORIZATION_HEADER, "Bearer " + authentication.getIAMProxyToken());
		if (StringUtils.isNotEmpty(authentication.getRXId()))
		{
			builder.header("rxId", authentication.getRXId());
		}
		if (StringUtils.isNotEmpty(authentication.getImpersonatedEmail()))
		{
			builder.header("X-CG-Impersonate", authentication.getImpersonatedEmail());
		}
		return builder;
	}

	protected int getInfoLength(InputStream responseData) throws IOException
	{
		ByteBuffer infoLengthBuffer = ByteBuffer.allocate(Integer.BYTES);
		infoLengthBuffer.put(responseData.readNBytes(Integer.BYTES));
		infoLengthBuffer.rewind();
		return infoLengthBuffer.getInt();
	}

	protected URI getURI(Authentication authentication)
	{
		try
		{
			String url = MessageFormat.format(BASE_PROVISION_URL,
					authentication.getCompany().getEnvironment() == Environment.Test ? DEVEL_PART : EMPTY,
					authentication.isSupport() ? SUPPORT_PART : EMPTY);
			//			String url = "http://localhost:8080/rx/provision";
			System.setProperty("candata.url", url);
			return new URL(url).toURI();
		}
		catch (MalformedURLException | URISyntaxException e)
		{
			throw new RuntimeException(e);
		}
	}

	protected void processResponse(Authentication authentication, MonitoredInputStream monitoredData, JSONObject info, ProgressMonitor monitor)
	{
		JSONObject properties = info.optJSONObject("properties");
		logger.log(Level.INFO, RX_ID_PROPERTY + ": " + properties.optString("instanceId"));
		System.setProperty(RX_ID_PROPERTY, properties.optString("instanceId"));
		authentication.setRXId(properties.optString("instanceId"));
		JSONArray bundles = info.optJSONArray("bundles");
		JSONArray toRemove = info.optJSONArray("toRemove");
		JSONArray clientData = info.optJSONArray("clientData");
		boolean removedBundles = remove(toRemove);
		File osgiDirectory = new File(System.getProperty("candata.workingDir"));
		File repositoryDirectory = new File(FolderUtils.getCandataDirectory(), "repository");
		repositoryDirectory.mkdir();
		int count = bundles.length();
		int total = getBundles(bundles)
				.map(ProvisionBundle::getSize)
				.reduce(0, Integer::sum)
				.blockingGet();
		monitor.configure("Updating application", total);
		logger.log(Level.INFO, "bundles to download: " + bundles.length() + " Size: " + total);

		AtomicInteger downloadedCount = new AtomicInteger();
		getBundles(bundles)
				.doOnNext(pb -> monitor.update(Progress.progress("Downloading " + downloadedCount.incrementAndGet() + " of " + count + " updates")))
				.map(b -> getBundle(repositoryDirectory, monitoredData, b))
				.observeOn(Schedulers.computation())
				.blockingSubscribe(b -> installBundle(osgiDirectory, b));
		configureClientData(clientData, monitoredData);
		monitor.configure("Starting application");
		if (removedBundles)
		{
			//				monitor.subTask("Removing Unnecessary Files");
			FrameworkWiring framework = ctx.getBundle(0).adapt(FrameworkWiring.class);
			framework.refreshBundles(null); // removed bundles
			//				monitor.worked(1);
		}
		logger.log(Level.INFO, "bundle count: " + Stream.of(ctx.getBundles())
				.count());
	}

	protected void configureClientData(JSONArray clientData, MonitoredInputStream monitoredData)
	{
		Map<String, InputStream> dataProviders = new HashMap<>();
		registerClientDataService(ctx, dataProviders);
		Observable.fromStream(StreamSupport.stream(clientData.spliterator(), false))
				.map(o -> (String) o)
				.blockingForEach(name -> configureClientData(monitoredData, name, dataProviders));
	}

	protected void configureClientData(MonitoredInputStream monitoredData, String name, Map<String, InputStream> dataProviders) throws IOException
	{
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		byte[] lengthBuffer = new byte[4];
		int read = monitoredData.read(lengthBuffer);
		int length = ByteBuffer.wrap(lengthBuffer)
				.getInt();
		while (length > 0)
		{
			out.write(monitoredData.readNBytes(length));
			read = monitoredData.read(lengthBuffer);
			if (read == -1)
			{
				length = 0;
			}
			else
			{
				length = ByteBuffer.wrap(lengthBuffer)
						.getInt();
			}
		}
		dataProviders.put(name, new ByteArrayInputStream(out.toByteArray()));
	}

	protected ServiceRegistration<Function> registerClientDataService(BundleContext ctx, Map<String, InputStream> dataProviders)
	{
		Dictionary<String, String> properties = new Hashtable<>();
		properties.put("name", "ClientDataProviderService");
		return ctx.registerService(Function.class, dataProviders::get, properties);
	}

	protected boolean remove(JSONArray toRemove)
	{
		List<String> names = getBundles(toRemove)
				.map(ProvisionBundle::getName)
				.toList()
				.blockingGet();
		Stream.of(ctx.getBundles())
				.filter(b -> !"org.eclipse.osgi".equals(b.getSymbolicName()))
				.filter(b -> !"com.candata.login".equals(b.getSymbolicName()))
				.filter(b -> names.contains(b.getSymbolicName()))
				.forEach(b -> {
					try
					{
						logger.log(Level.INFO, "uninstalling " + b.getSymbolicName());
						b.uninstall();
					}
					catch (BundleException e)
					{
						error("error uninstalling bundle " + b.getSymbolicName(), e);
					}
				});
		return !names.isEmpty();
	}

	protected Observable<ProvisionBundle> getBundles(JSONArray bundles)
	{
		if (bundles == null)
		{
			return Observable.empty();
		}
		return Observable.fromStream(StreamSupport.stream(bundles.spliterator(), false))
				.map(o -> (JSONObject) o)
				.map(ProvisionBundle::new)
				.sorted(Comparator.comparing(ProvisionBundle::getSequence));
	}

	protected File getBundle(File repositoryDirectory, MonitoredInputStream monitoredData, ProvisionBundle bundle)
	{
		File bundleFile = new File(repositoryDirectory, bundle.getName());
		try (FileOutputStream fos = new FileOutputStream(bundleFile))
		{
			logger.log(Level.INFO, "downloading: " + bundle.getSequence() + " " + bundle.getName() + " (" + bundle.getSize() + ")");
			monitoredData.transferTo(fos, bundle.getSize());
			return bundleFile;
		}
		catch (IOException e)
		{
			throw new RuntimeException("error reading bundle", e);
		}
	}

	protected void installBundle(File repositoryDir, File bundleFile)
	{
		try
		{
			if (bundleFile.getName().equals("org.eclipse.swt"))
			{
				return;
			}
			if (isGzipped(bundleFile))
			{
				handleVB6(repositoryDir, bundleFile);
				return;
			}
			//			monitor.subTask("Installing Updates");
			//			error("Installing bundle " + bundleFile.getName());
			Bundle bundle = ctx.installBundle("reference:file:" + bundleFile.getAbsolutePath());
			error("   Installed bundle " + bundle.getSymbolicName());
			bundle.adapt(BundleStartLevel.class).setStartLevel(UI_START_LEVEL);
			if (!isFragment(bundle))
			{
				//				monitor.subTask("Starting Application");
				bundle.start(Bundle.START_ACTIVATION_POLICY);
				error("   Started bundle " + bundle.getSymbolicName());
			}
			else
			{
				error("   Skipped fragment " + bundle.getSymbolicName());
			}
		}
		catch (BundleException e)
		{
			error("Error adding bundle " + bundleFile.getName(), e);
			throw new RuntimeException(e);
		}
	}

	protected void handleVB6(File repositoryDir, File downloadedFiled)
	{
		try
		{
			processVb6(getVb6RepoDir(repositoryDir), downloadedFiled);
		}
		catch (IOException e)
		{
			error("Error installing vb6  " + downloadedFiled, e);
		}
	}

	protected void processVb6(File vb6Repository, File downloadedFiled) throws FileNotFoundException, IOException
	{
		String name = downloadedFiled.getName();
		File uncompressed = new File(vb6Repository, name.substring(0, name.length() - 3));
		if (uncompressed.getName().endsWith("exe"))
		{
			System.setProperty("vb6.executable", uncompressed.getAbsolutePath());
		}
		if (!vb6Repository.exists())
		{
			vb6Repository.mkdir();
		}
		if (!uncompressed.exists())
		{
			Files.copy(new GZIPInputStream(new FileInputStream(downloadedFiled)), uncompressed.toPath());
		}
	}

	protected File getVb6RepoDir(File repositoryDir)
	{
		return new File(repositoryDir, "vb6");
	}

	private boolean isGzipped(File bundleFile)
	{
		return bundleFile.getName().toLowerCase().endsWith(GZIP_EXTENSION);
	}

	private boolean isFragment(Bundle bundle)
	{
		return (bundle.adapt(BundleRevision.class).getTypes() & BundleRevision.TYPE_FRAGMENT) != 0;
	}

	protected JSONObject addInfo(JSONArray bundles)
	{
		JSONObject info = new JSONObject();
		info.put("installerVersion", Version.VERSION);
		info.put("osName", System.getProperty("os.name"));
		info.put("osVersion", System.getProperty("os.version"));
		OperatingSystemMXBean mxBean = ManagementFactory.getOperatingSystemMXBean();
		info.put("arch", mxBean.getArch());
		info.put("bundles", bundles);
		return info;
	}

	protected JSONObject toJSON(Bundle bundle)
	{
		JSONObject asJSON = new JSONObject();
		asJSON.put("name", bundle.getSymbolicName());
		asJSON.put("version", bundle.getVersion());
		return asJSON;
	}

	protected void error(String message)
	{
		error(message, null);
	}

	protected void error(String message, Throwable t)
	{
		System.err.println(message);
		logger.log(Level.SEVERE, message, t);
	}

	protected static java.util.logging.Logger logger = java.util.logging.Logger.getLogger("Candata");

	BundleContext ctx;
	static final String PIPE = "|";
	static final String NEW_LINE = "\n";
	public static final int START_LEVEL = 6;
	public static final int UI_START_LEVEL = 10;
	static final String BASE_PROVISION_URL = "https://app.{0}candata.com/rx/{1}provision";
	static final String DEVEL_PART = "devel.";
	static final String SUPPORT_PART = "support/";
	static final String GZIP_EXTENSION = ".gz";
}
