package com.candata.login.zoo.widgets;

/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Mark Siegel <mark.siegel@businessobjects.com> - Fix for Bug 184533
 *              [Progress] ProgressIndicator uses hardcoded style for ProgressBar
 *******************************************************************************/

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StackLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.TaskBar;
import org.eclipse.swt.widgets.TaskItem;

/**
 * A control for showing progress feedback for a long running operation. This
 * control supports both determinate and indeterminate SWT progress bars. For
 * indeterminate progress, we don't have to know the total amount of work in
 * advance and no <code>worked</code> method needs to be called.
 */
public class ProgressIndicator extends Composite
{
	//	private final static int PROGRESS_MAX = 1000; // value to use for max in

	// progress bar
	private boolean animated = true;

	private final StackLayout layout;

	private final ProgressBar determinateProgressBar;

	private final ProgressBar indeterminateProgressBar;

	private double totalWork;

	private double sumWorked;

	/**
	 * Create a ProgressIndicator as a child under the given parent.
	 *
	 * @param parent
	 *            The widgets parent
	 */
	public ProgressIndicator(Composite parent)
	{
		this(parent, SWT.SMOOTH);
	}

	/**
	 * Create a ProgressIndicator as a child under the given parent.
	 *
	 * @param parent
	 *            The widgets parent
	 * @param style
	 *            the SWT style constants for progress monitors created by the
	 *            receiver.
	 * @since 3.4
	 */
	public ProgressIndicator(Composite parent, int style)
	{
		super(parent, SWT.NULL);

		// Enforce horizontal only if vertical isn't set
		if ((style & SWT.VERTICAL) == 0)
		{
			style |= SWT.HORIZONTAL;
		}

		determinateProgressBar = new ProgressBar(this, style);
		indeterminateProgressBar = new ProgressBar(this, style | SWT.INDETERMINATE);
		layout = new StackLayout();
		setLayout(layout);
		taskItem = getTaskItem();
	}

	/**
	 * Initialize the progress bar to be animated.
	 */
	public void beginAnimatedTask()
	{
		done();
		layout.topControl = indeterminateProgressBar;
		layout();
		configureTaskBar(-1);
		animated = true;
	}

	/**
	 * Initialize the progress bar.
	 *
	 * @param max
	 *            The maximum value.
	 */
	public void beginTask(int max)
	{
		done();
		this.totalWork = max;
		this.sumWorked = 0;
		determinateProgressBar.setMinimum(0);
		determinateProgressBar.setMaximum(max);
		determinateProgressBar.setSelection(0);
		layout.topControl = determinateProgressBar;
		layout();
		configureTaskBar(max);
		animated = false;
	}

	/**
	 * Progress is done.
	 */
	public void done()
	{
		if (determinateProgressBar.isDisposed())
		{
			taskBarDone();
			return;
		}
		if (!animated)
		{
			determinateProgressBar.setMinimum(0);
			determinateProgressBar.setMaximum(0);
			determinateProgressBar.setSelection(0);
		}
		layout.topControl = null;
		taskBarDone();
		if (isDisposed())
		{
			return;
		}
		layout();
	}

	/**
	 * Moves the progress indicator to the end.
	 */
	public void sendRemainingWork()
	{
		worked(totalWork - sumWorked);
	}

	/**
	 * Moves the progress indicator by the given amount of work units
	 *
	 * @param work
	 *            the amount of work to increment by.
	 */
	public void worked(double work)
	{
		if (work == 0 || animated)
		{
			return;
		}
		sumWorked = work;
		if (sumWorked > totalWork)
		{
			sumWorked = totalWork;
		}
		if (sumWorked < 0)
		{
			sumWorked = 0;
		}
		int value = Double.valueOf(sumWorked).intValue();
		if (determinateProgressBar.getSelection() < value)
		{
			determinateProgressBar.setSelection(value);
		}
		setTaskBarProgress(sumWorked);
	}

	/**
	 * Show the receiver as showing an error.
	 *
	 * @since 3.4
	 */
	public void showError()
	{
		determinateProgressBar.setState(SWT.ERROR);
		indeterminateProgressBar.setState(SWT.ERROR);
	}

	/**
	 * Show the receiver as being paused.
	 *
	 * @since 3.4
	 */
	public void showPaused()
	{
		determinateProgressBar.setState(SWT.PAUSED);
		indeterminateProgressBar.setState(SWT.PAUSED);
	}

	/**
	 * Reset the progress bar to it's normal style.
	 *
	 * @since 3.4
	 */
	public void showNormal()
	{
		determinateProgressBar.setState(SWT.NORMAL);
		indeterminateProgressBar.setState(SWT.NORMAL);
	}

	protected void configureTaskBar(int totalWork)
	{
		if (taskItem != null && !taskItem.isDisposed())
		{
			taskItem.setProgressState((totalWork == -1) ? SWT.INDETERMINATE : SWT.NORMAL);
		}
	}

	protected void setTaskBarProgress(double worked)
	{
		if (taskItem != null && !taskItem.isDisposed() && totalWork > 0)
		{
			taskItem.setProgressState(SWT.DEFAULT);
			taskItem.setProgress(Float.valueOf(Double.valueOf(worked).floatValue() / Double.valueOf(totalWork).floatValue() * 100).intValue());
		}
	}

	protected void taskBarDone()
	{
		if (taskItem != null && !taskItem.isDisposed())
		{
			taskItem.setProgressState(SWT.DEFAULT);
		}
	}

	protected TaskItem getTaskItem()
	{
		TaskBar taskBar = determinateProgressBar.getDisplay().getSystemTaskBar();
		if (taskBar != null)
		{
			TaskItem taskItem = taskBar.getItem(determinateProgressBar.getShell());
			if (taskItem == null)
			{
				taskItem = taskBar.getItem(null);
			}
			return taskItem;
		}
		return null;
	}

	protected TaskItem taskItem;
}