package com.candata.login.utils;

import java.net.HttpURLConnection;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpRequest.BodyPublisher;
import java.net.http.HttpRequest.Builder;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandler;
import java.net.http.HttpResponse.BodyHandlers;
import java.util.List;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;

import io.reactivex.rxjava3.core.Maybe;
import io.reactivex.rxjava3.core.Single;
import io.reactivex.rxjava3.subjects.MaybeSubject;

public class HttpUtils
{
	public enum ContentType
	{
		JSON("application/json; charset=utf-8"), FORM("application/x-www-form-urlencoded");

		private ContentType(String type)
		{
			this.type = type;
		}

		public String getType()
		{
			return type;
		}

		private String type;
	}

	public static Single<Builder> buildGet(URI uri, String token)
	{
		return Single.fromCallable(() -> HttpRequest.newBuilder(uri))
				.map(b -> b.header(AUTHORIZATION_HEADER, BEARER_KEY + token))
				.map(Builder::GET);
	}

	public static Single<Builder> buildPost(URI uri, ContentType contentType, BodyPublisher publisher)
	{
		return Single.fromCallable(() -> HttpRequest.newBuilder(uri))
				.map(b -> b.header(CONTENT_TYPE_HEADER, contentType.getType()))
				.map(b -> b.POST(publisher));
	}

	public static Single<Builder> buildPost(URI uri, ContentType contentType, BodyPublisher publisher, String token)
	{
		return buildPost(uri, contentType, publisher)
				.map(b -> b.header(AUTHORIZATION_HEADER, BEARER_KEY + token));
	}

	public static <T> Maybe<T> doRequest(HttpRequest request, BodyHandler<T> handler)
	{
		return MaybeSubject.create(e -> {
			HttpClient client = HttpClient.newHttpClient();
			try
			{
				client.sendAsync(request, handler)
						.thenAccept(r -> {
							int status = r.statusCode();
							if (status != HttpURLConnection.HTTP_OK)
							{
								if (!e.isDisposed())
								{
									System.err.println("error processing request (" + status + ") " + r.body() + "\nfor request: " + request);
									e.onError(new RuntimeException(
											"error processing request (" + status + ") " + r.body() + "\nfor request: " + request));
								}
								return;
							}
							if (!e.isDisposed())
							{
								e.onSuccess(r.body());
							}
						})
						.orTimeout(10, TimeUnit.SECONDS)
						.join();
			}
			catch (Exception ex)
			{
				if (!e.isDisposed())
				{
					e.onError(ex);
				}
			}
		});
	}

	public static Maybe<String> doRequest(HttpRequest request)
	{
		return doRequest(request, BodyHandlers.ofString());
	}

	public static String headers(HttpResponse<?> response)
	{
		StringBuilder headers = new StringBuilder();
		for (Entry<String, List<String>> entry : response.headers().map().entrySet())
		{
			for (String value : entry.getValue())
			{
				headers.append(entry.getKey());
				headers.append(": ");
				headers.append(value);
				headers.append("\n");
			}
		}
		return headers.toString();
	}

	public static String headers(HttpRequest request)
	{
		StringBuilder headers = new StringBuilder();
		for (Entry<String, List<String>> entry : request.headers().map().entrySet())
		{
			for (String value : entry.getValue())
			{
				headers.append(entry.getKey());
				headers.append(": ");
				headers.append(value);
				headers.append("\n");
			}
		}
		return headers.toString();
	}

	public static final String CONTENT_TYPE_HEADER = "Content-Type";
	public static final String AUTHORIZATION_HEADER = "Authorization";
	private static final String BEARER_KEY = "Bearer ";
}