package com.candata.login.oauth.oauth2;

import static com.candata.login.utils.StringUtils.EMPTY;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import com.candata.login.oauth.beans.Authentication;
import com.candata.login.oauth.beans.Authentication.SupportAuthentications;
import com.candata.login.oauth.beans.FlowProperties;
import com.candata.login.oauth.oauth2.beans.OAuthProperties;
import com.candata.login.oauth.support.oauth2.beans.OAuthSupportProperties.Flows;
import com.candata.login.oauth.zoo.oauth2.AuthenticationBrowser;
import com.candata.login.oauth.zoo.oauth2.AuthenticationParameters;
import com.candata.login.oauth.zoo.support.oauth2.IdCodeInstalledApp;
import com.candata.login.oauth.zoo.support.oauth2.beans.IdCredential;
import com.candata.login.oauth.zoo.support.oauth2.store.TokenDataStoreFactory;
import com.candata.login.utils.DefaultTrustManager;
import com.candata.login.utils.FolderUtils;
import com.candata.login.utils.Platform;
import com.google.api.client.auth.oauth2.AuthorizationCodeFlow;
import com.google.api.client.auth.oauth2.BearerToken;
import com.google.api.client.extensions.jetty.auth.oauth2.LocalServerReceiver;
import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.javanet.NetHttpTransport;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.jackson2.JacksonFactory;

import io.reactivex.rxjava3.core.Maybe;
import io.reactivex.rxjava3.schedulers.Schedulers;

public class SupportAuthenticator
{
	/** Directory to store user credentials. */
	private static final File DATA_STORE_DIR = new File(FolderUtils.getCandataDirectory(), "/" + (Platform.isWindows() ? "auth" : ".auth"));

	/**
	 * Global instance of the {@link DataStoreFactory}. The best practice is to make it a single
	 * globally shared instance across your application.
	 */
	//	private static FileDataStoreFactory DATA_STORE_FACTORY;

	/** Global instance of the HTTP transport. */
	private static final HttpTransport HTTP_TRANSPORT = new NetHttpTransport();

	/** Global instance of the JSON factory. */
	static final JsonFactory JSON_FACTORY = new JacksonFactory();

	private static final String OPEN_ID_SCOPE = "openid";
	private static final String EMAIL_SCOPE = "https://www.googleapis.com/auth/userinfo.email";
	private static final String PROFILE_SCOPE = "https://www.googleapis.com/auth/userinfo.profile";

	protected static Authentication currentAuthentication;
	static
	{
		DefaultTrustManager.configure();
	}

	public static Authentication getAuthentication()
	{
		return currentAuthentication;
	}

	public static Maybe<SupportAuthentications> authenticate(Flows flows)
	{
		SupportAuthentications authentications = new SupportAuthentications();
		return authenticate(flows.testProperties())
				.map(authentications::setTest)
				.ignoreElement()
				.andThen(authenticate(flows.productionProperties()))
				.map(authentications::setProduction);
	}

	public static Maybe<Authentication> authenticate(FlowProperties properties)
	{
		return Maybe.fromCallable(() -> new SupportAuthenticator().doAuthenticate(properties))
				.subscribeOn(Schedulers.computation());
	}

	protected File getDataStore(File baseDirectory, FlowProperties properties)
	{
		boolean isTest = properties.audience().equals(OAuthProperties.TEST_IAM_PROXY_AUDIENCE);
		File directory = new File(baseDirectory, "candata" + (isTest ? "-test" : EMPTY));
		return new File(directory, properties.email());
	}

	protected Authentication doAuthenticate(FlowProperties properties) throws IOException
	{
		boolean isTest = properties.audience().equals(OAuthProperties.TEST_IAM_PROXY_AUDIENCE);
		// set up authorization code flow
		AuthorizationCodeFlow flow = new AuthorizationCodeFlow.Builder(BearerToken
				.authorizationHeaderAccessMethod(), HTTP_TRANSPORT, JSON_FACTORY,
				new GenericUrl(properties.tokenURL().toString()),
				new AuthenticationParameters(properties.clientId(), properties.secret())
						.setAudience(properties.audience()),
				properties.clientId(), properties.authURL().toString())
						.setScopes(getScopes(isTest))
						.setDataStoreFactory(new TokenDataStoreFactory(properties, getDataStore(DATA_STORE_DIR, properties)))
						.build();
		LocalServerReceiver receiver = new LocalServerReceiver();
		IdCredential credential = new IdCodeInstalledApp(flow, receiver, new AuthenticationBrowser(() -> onCancel(receiver), SUCCESS_PAGE_TEXT))
				.authorize();
		if (credential == null)
		{
			return null;
		}
		Authentication auth = new Authentication(credential);
		auth.setCompany(properties.company());
		confirmEmail(properties, auth);
		auth.setRXId(System.getProperty(Constants.RX_ID_PROPERTY));
		currentAuthentication = auth;
		return auth;
	}

	protected void confirmEmail(FlowProperties properties, Authentication auth)
	{
		String authenticatedAs = auth.getIdTokenPayload().optString("email");
		if (!properties.email().equals(authenticatedAs))
		{
			clearFailedCredential(properties);
			throw new RuntimeException("Authenticated email address is different from original.");
		}
	}

	protected void clearFailedCredential(FlowProperties properties)
	{
		File datastore = getDataStore(DATA_STORE_DIR, properties);
		if (datastore.exists())
		{
			if (!datastore.delete())
			{
				File idFile = new File(datastore, "ID");
				if (idFile.exists())
				{
					idFile.delete();
				}
			}
		}
	}

	protected List<String> getScopes(boolean isTest)
	{
		if (isTest)
		{
			return Arrays.asList(OPEN_ID_SCOPE, EMAIL_SCOPE);
		}
		return Arrays.asList(OPEN_ID_SCOPE, EMAIL_SCOPE, PROFILE_SCOPE);
	}

	protected void onCancel(LocalServerReceiver receiver)
	{
		try
		{
			receiver.stop();
		}
		catch (IOException e)
		{
		}
	}

	private static final String SUCCESS_PAGE_TEXT = "Received verification code";
	public static final String AUTHENTICATION_KEY = "CG-AUTHENTICATION";
}
