package com.candata.login.oauth.oauth2;

import java.io.File;
import java.util.Arrays;
import java.util.List;

import com.candata.login.oauth.beans.Authentication;
import com.candata.login.oauth.beans.FlowProperties;
import com.candata.login.oauth.oauth2.beans.OAuthProperties;
import com.candata.login.oauth.users.Company;
import com.candata.login.oauth.zoo.support.oauth2.google.TokenExchange;

import io.reactivex.rxjava3.core.Maybe;
import io.reactivex.rxjava3.schedulers.Schedulers;

public class UserAuthenticator extends SupportAuthenticator
{
	private static final String OPEN_ID_SCOPE = "openid";
	private static final String EMAIL_SCOPE = "email";
	private static final String PROFILE_SCOPE = "profile";
	private static final String OFFLINE_ACCESS_SCOPE = "offline_access";

	public static Maybe<Authentication> authenticate(FlowProperties properties)
	{
		return Maybe.fromCallable(() -> new UserAuthenticator().doAuthenticate(properties))
				.subscribeOn(Schedulers.computation())
				.flatMap(auth -> convert(auth, properties.company()));
	}

	@Override
	protected File getDataStore(File baseDirectory, FlowProperties properties)
	{
		File directory = new File(baseDirectory, properties.company().getRealm());
		return new File(directory, properties.email());
	}

	@Override
	protected List<String> getScopes(boolean isTest)
	{
		return Arrays.asList(OPEN_ID_SCOPE, EMAIL_SCOPE, PROFILE_SCOPE, OFFLINE_ACCESS_SCOPE);
	}

	protected static Maybe<Authentication> convert(Authentication authentication, Company company)
	{
		return TokenExchange.exchangeForTemp(OAuthProperties.getAudience(company), authentication.getAccessToken())
				.map(sts -> sts.getString(ACCESS_TOKEN_KEY))
				.flatMap(token -> TokenExchange.exchangeForAccess(OAuthProperties.getServiceAccount(company),
						OAuthProperties.getIAMProxyAudience(company), token))
				.map(result -> authentication.setIAMProxyToken(result.getString(TOKEN_KEY)))
				.doOnSuccess(auth -> auth.setCompany(company))
				.doOnSuccess(auth -> currentAuthentication = auth);
	}

	private static final String ACCESS_TOKEN_KEY = "access_token";
	private static final String TOKEN_KEY = "token";
}
