package com.candata.login.oauth.zoo.support.oauth2.store;

import static com.candata.login.oauth.zoo.security.Encrypt.decrypt;
import static com.candata.login.oauth.zoo.security.Encrypt.encrypt;
import static java.nio.file.StandardOpenOption.TRUNCATE_EXISTING;
import static java.nio.file.StandardOpenOption.WRITE;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.time.OffsetDateTime;
import java.util.HashMap;
import java.util.Map;

import org.json.JSONObject;

import com.candata.login.oauth.zoo.support.oauth2.beans.IdCredential;
import com.google.api.client.auth.oauth2.StoredCredential;
import com.google.api.client.util.store.AbstractDataStoreFactory;
import com.google.api.client.util.store.DataStore;
import com.google.api.client.util.store.FileDataStoreFactory;

public class IdStore
{
	public IdStore(DataStore<StoredCredential> dataStore)
	{
		this.dataStoreFactory = (AbstractDataStoreFactory) dataStore.getDataStoreFactory();
	}

	public IdCredential load()
	{
		try
		{
			if (dataStoreFactory instanceof FileDataStoreFactory)
			{
				File idFile = new File(((FileDataStoreFactory) dataStoreFactory).getDataDirectory(), "ID");
				if (idFile.exists())
				{
					updateStore(new JSONObject(decrypt(Files.readString(idFile.toPath()))));
				}
			}
		}
		catch (IllegalArgumentException e)
		{
			//just carry on
		}
		catch (IOException e)
		{
			throw new RuntimeException("failed to read ID store", e);
		}
		IdCredential credential = new IdCredential();
		credential.setRefreshToken(store.get(REFRESH_TOKEN_KEY));
		credential.setAccessToken(store.get(ACCESS_TOKEN_KEY));
		credential.setIdToken(store.get(ID_TOKEN_KEY));

		long expiresIn = 0;
		if (!store.getOrDefault(EXPIRES_AT_KEY, "").equals(""))
		{
			OffsetDateTime expiresAt = OffsetDateTime.parse(store.getOrDefault(EXPIRES_AT_KEY, ""));
			expiresIn = expiresAt.toEpochSecond() - OffsetDateTime.now().toEpochSecond();
		}
		credential.setExpiresInSeconds(expiresIn);
		return credential;
	}

	public IdCredential save(IdCredential credential)
	{
		updateStore(credential);
		save();
		return credential;
	}

	protected void save()
	{
		if (!(dataStoreFactory instanceof FileDataStoreFactory))
		{
			return;
		}
		try
		{
			File idFile = new File(((FileDataStoreFactory) dataStoreFactory).getDataDirectory(), "ID");
			idFile.createNewFile();
			JSONObject data = new JSONObject();
			data.put(REFRESH_TOKEN_KEY, store.get(REFRESH_TOKEN_KEY));
			data.put(ACCESS_TOKEN_KEY, store.get(ACCESS_TOKEN_KEY));
			data.put(ID_TOKEN_KEY, store.get(ID_TOKEN_KEY));
			data.put(EXPIRES_AT_KEY, store.get(EXPIRES_AT_KEY));

			Files.writeString(idFile.toPath(), encrypt(data.toString()), TRUNCATE_EXISTING, WRITE);
		}
		catch (IOException e)
		{
			throw new RuntimeException("failed to create ID store", e);
		}
	}

	protected void updateStore(JSONObject data)
	{
		Long expiresIn = OffsetDateTime.parse(data.getString(EXPIRES_AT_KEY)).toEpochSecond() - OffsetDateTime.now().toEpochSecond();
		updateStore(data.optString(REFRESH_TOKEN_KEY), data.getString(ACCESS_TOKEN_KEY), data.getString(ID_TOKEN_KEY), expiresIn);
	}

	protected void updateStore(IdCredential credential)
	{
		updateStore(credential.getRefreshToken(), credential.getAccessToken(), credential.getIdToken(), credential.getExpiresInSeconds());
	}

	protected void updateStore(String refresh, String access, String id, long expiresIn)
	{
		store.put(REFRESH_TOKEN_KEY, refresh);
		store.put(ACCESS_TOKEN_KEY, access);
		store.put(ID_TOKEN_KEY, id);
		store.put(EXPIRES_AT_KEY, OffsetDateTime.now().plusSeconds(expiresIn).toString());
	}

	AbstractDataStoreFactory dataStoreFactory;
	static final String REFRESH_TOKEN_KEY = "refreshToken";
	static final String ACCESS_TOKEN_KEY = "accessToken";
	static final String ID_TOKEN_KEY = "idToken";
	static final String EXPIRES_AT_KEY = "expiresAt";

	static final Map<String, String> store = new HashMap<>();
}
